# -*- coding: utf-8 -*-

from flask import request

from blueflask.lib.errors import forbidden, not_found
from blueflask.lib.infos import success

from ....consts import ADDRESS, API_REGISTRY, HOSTNAME, SERVICE_CODE
from . import api

import redis


@api.route('/<string:env>/registry/<string:service>', methods=['GET'])
@api.route('/registry/<string:service>', methods=['GET'])
def get_service_infos_in_registry(service, env=None):
    """
    Informations sur un service dans le registre.
    ---
    tags:
      - Registry
    """
    host = None
    source = request.headers.get('X-Real-Ip')

    if source not in API_REGISTRY and source != ADDRESS:
        # si la requête provient d'une IP qui n'héberge pas un service
        # on interdit l'accès au registre
        return forbidden(
            service_code=SERVICE_CODE
            )

    if env and not HOSTNAME == 'api':
        # si la requête ressemble à /dev/registry ou /staging/registry
        # et qu'elle ne provient pas de l'hôte api.blueline.mg
        # on retourne un 404
        return not_found(service_code=SERVICE_CODE)
    elif env == 'dev':
        host = 'bldv040v.blueline.mg'
    elif env == 'staging':
        host = 'blst040v.blueline.mg'

    if not host:
        # on regarde le registre local
        infos = API_REGISTRY.hgetall('service:{}'.format(service))
    else:
        # on regarde le registre distant
        registry = redis.Redis(host=host+'.blueline.mg', db=15)
        infos = registry.hgetall('service:{}'.format(service))

    if not infos:
        return not_found(service_code=SERVICE_CODE)

    result = {
        key.decode('utf-8'): value.decode('utf-8')
        for key, value in infos.items()
        }
    return success(service_code=SERVICE_CODE, message=result)


@api.route('/<string:env>/registry', methods=['POST', 'DELETE'])
@api.route('/registry', methods=['POST', 'DELETE'])
def set_info_in_registry(env=None):
    """
    Inscription d'un service au registre.
    ---
    tags:
      - Registry
    """
    data = request.get_json()
    service = data['service']
    key = 'service:{}'.format(service)
    host = None
    source = request.headers.get('X-Real-Ip')

    if source not in API_REGISTRY or source != ADDRESS:
        # si la requête provient d'une IP qui n'héberge pas un service
        # on interdit l'accès au registre
        return forbidden(service_code=SERVICE_CODE)

    if env and not HOSTNAME == 'api':
        # si la requête ressemble à /dev/registry ou /staging/registry
        # et qu'elle ne provient pas de l'hôte api.blueline.mg
        # on retourne un 404
        return not_found(service_code=SERVICE_CODE)
    elif env == 'dev':
        host = 'bldv040v.blueline.mg'
    elif env == 'staging':
        host = 'blst040v.blueline.mg'

    if not host:
        # on regarde le registre local
        registry = API_REGISTRY
    else:
        # on regarde le registre distant
        registry = redis.Redis(host=host+'.blueline.mg', db=15)

    if request.method == 'DELETE':
        if key not in [i.decode('utf-8') for i in registry.keys()]:
            return not_found(
                message="Ce service n'existe pas",
                service_code=SERVICE_CODE
                )
        registry.delete(key)
    else:
        address = data['address']
        del data['address']
        registry.delete(key)
        registry.hmset(key, data)
        registry.set(address, data['host'])
    return success(service_code=SERVICE_CODE)

# EOF
