# -*- coding: utf-8 -*-

from flask import Flask, g, jsonify, request
from flask_swagger import swagger
from flask_swagger_ui import get_swaggerui_blueprint
from systemd import journal

from blueflask.lib.errors import forbidden
from blueflask.lib.errors import internal_error
from blueflask.lib.errors import not_allowed
from blueflask.lib.errors import not_found
from blueflask.lib.errors import unauthorized

from .services import api
from .services import __version__
from ... import LOGS


import uuid


__all__ = [
    'app'
    ]

app = Flask(__name__)
app.config['LOGGER_NAME'] = 'sponge-api'
app.logger.handlers = LOGS.logger.handlers

swag_blueprint = get_swaggerui_blueprint('/apidoc', '/api/spec')
app.register_blueprint(swag_blueprint, url_prefix='/apidoc')
app.register_blueprint(api)


@app.errorhandler(401)
def unauthorized_error(e):
    return unauthorized(
        request_id=request.headers.get('X-Request-Id'),
        code='04-401'
        )


@app.errorhandler(403)
def forbidden_error(e):
    return forbidden(
        request_id=request.headers.get('X-Request-Id'),
        code='04-403'
        )


@app.errorhandler(404)
def not_found_error(e):
    return not_found(
        request_id=request.headers.get('X-Request-Id'),
        code='04-404'
        )


@app.errorhandler(405)
def method_not_allowed_error(e):
    return not_allowed(
        request_id=request.headers.get('X-Request-Id'),
        code='04-405'
        )


@app.errorhandler(500)
def internal_server_error(e):
    return internal_error(
        request_id=request.headers.get('X-Request-Id'),
        code='04-500'
        )


def actions_before_request():
    """Création d'un callback permettant d'exécuter du code
    avant celui du endpoint appelé"""
    headers = {  # on copie les en-têtes de la requête
        key: value for key, value in request.headers.items()
        }
    if not request.headers.get('X-Request-Id'):
        # si la requête n'a pas d'identifiant on en génère un au hasard
        g.request_id = str(uuid.uuid4())
        headers['X-Request-Id'] = g.request_id
    else:
        g.request_id = request.headers.get('X-Request-Id')
    request.headers = headers
    msg_debug = (
        'Request-Id:{req_id} '
        '{method} '
        '{url} '
        'params={params} '
        'json={json_data}'
        ).format(
            req_id=g.request_id,
            method=request.method,
            url=request.base_url,
            params=request.args or {},
            json_data=request.get_json() or {}
            )
    # on essaye de récupérer le nom de l'utilisateur qui effectue la requête
    try:
        user = g.user
    except Exception:
        user = None
    else:
        if isinstance(user, list):
            user = min(user, key=lambda x: len(x))
    if user:
        msg_debug = 'User:{user} {msg}'.format(user=user, msg=msg_debug)
    journal.send(  # on envoie les détails au journal de systemd
        msg_debug,
        REQUEST_ID=g.request_id
        )


def actions_after_request(response):
    """Création d'un callback permettant d'exécuter du code
    après que la requête soit achevée"""
    response.headers['X-Request-Id'] = g.request_id
    msg_debug = (
        'Request-Id:{req_id} '
        'Status:{status} '
        'Response:{response}').format(
            req_id=g.request_id,
            status=response.status_code,
            response=response.response
            )
    try:
        user = g.user
    except Exception:
        user = None
    else:
        if isinstance(user, list):
            user = min(user, key=lambda x: len(x))
    if user:
        msg_debug = 'User:{user} {msg}'.format(user=user, msg=msg_debug)
    journal.send(
        msg_debug,
        REQUEST_ID=g.request_id,
        STATUS=response.status_code
        )
    return response


@app.route('/api/spec')
def spec():
    """Création de la documentation OpenAPI au format JSON"""
    swag = swagger(app)
    swag['info']['title'] = 'Sponge'
    swag['info']['description'] = "API Gateway"
    swag['info']['version'] = __version__
    return jsonify(swag)


app.before_request(actions_before_request)
app.after_request(actions_after_request)


if __name__ == '__main__':
    app.run(debug=True, port=1235)

# EOF
