#!/usr/bin/env python
# encoding: utf-8


from flask import jsonify, request

from blueflask.lib.infos import HTTPInfo
from blueflask.lib.errors import HTTPError
from blueflask.lib.errors import not_found
from blueflask.lib.errors import unauthorized

from ....consts import API_REGISTRY
from ....consts import LDAP_API
from ....consts import HOSTNAME
from ....consts import SERVICE_CODE
from .... import gateway
from . import api

from rest_client import RestClient


@api.route(
    '/<string:env>/<string:service>/v<int:version>/<path:uri>',
    methods=['GET', 'POST', 'PUT', 'DELETE']
    )
@api.route(
    '/<string:service>/v<int:version>/<path:uri>',
    methods=['GET', 'POST', 'PUT', 'DELETE']
    )
def api_gateway(service, version, uri, env=None):
    """
    API Gateway.
    ---
    tags:
      - Gateway
    """
    auth = request.authorization
    source = request.headers['X-Real-Ip']
    password = auth.password
    username = auth.username
    if source not in API_REGISTRY:
        usernames = verify_token(request, username, service, uri)
        if not auth or not usernames:
            return unauthorized(code='04-401')
        elif not isinstance(usernames, bool):
            username = min(usernames, key=lambda x: len(x))
            if len(username) > 20:
                username = username[:20]
    params = request.args or {}
    data = request.form or {}
    json_doc = request.get_json() or {}
    host = None
    if env and not HOSTNAME.startswith('api'):
        return not_found(code='04-404')
    elif env == 'dev':
        host = 'bldv040v'
    elif env == 'staging':
        host = 'blst040v'
    gw = gateway.Api(service, host=host, version=version)
    result = gw.call(
        request.method,
        '/{}'.format(uri),
        headers=request.headers,
        auth=(username, password),
        params=params,
        data=data,
        json=json_doc
        )
    status = result['status']
    content = result['content']
    if isinstance(result['content'], str):
        if status >= 400:
            response = HTTPError(
                status,
                service_code=SERVICE_CODE
                ).response(content)
        else:
            response = HTTPInfo(
                status,
                service_code=SERVICE_CODE
                ).response(content)
    else:
        response = jsonify(content)
        response.status_code = status
    return response


def verify_token(request, token, service, uri):
    """Vérifie avec le serveur LDAP si le token d'authentification
    est toujours valide.

    La fonction vérifie d'abord dans le registre d'APIs
    si le token utilisé n'a pas déjà été vérifié par un service de confiance.
    Si ce n'est pas le cas, alors une vérification est effectuée auprès de
    l'annuaire LDAP et le résultat est stocké dans le registre d'APIs."""

    ldap_client = RestClient(LDAP_API)

    if service == 'ldap' and uri.startswith('auth'):
        return True  # on laisse le endpoint gérer la requête

    key = 'token:{}'.format(token)
    result = API_REGISTRY.get(key)
    if not result:  # on demande une vérification de token
        ldap_client = RestClient(LDAP_API)
        response = ldap_client.post(
            '/auth/verify-token',
            auth=(token, ''),
            headers=request.headers
            )
        if response['status'] != 200:  # token invalide
            return False
        else:
            # on récupère les noms d'utilisateurs correspondant à ce token
            user = response['content']['message']
            API_REGISTRY.set(key, user)
            API_REGISTRY.expire(key, 3600)
            # on force l'enregistrement dans redis à expirer
            # au bout d'une heure
            return user
    else:
        # le token a déjà été vérifié une fois
        user = result.decode('utf-8')
        return eval(user)

# EOF
