#!/usr/bin/env python
# encoding: utf-8

from systemd import journal

from .consts import APIGW, CONFIG, LOGS, SERVICE_CODE
from .core import Sms

import datetime
import json
import random
import threading
import time

import pika
import requests


def process(queue):
    """Traitement des files d'attentes"""

    def callback(ch, method, properties, body):
        try:
            document = json.loads(body.decode('utf-8'))
        except Exception as exc:
            LOGS.logger.error(exc)
            ch.basic_ack(delivery_tag=method.delivery_tag)
        else:
            http_verb = document['method']
            http_uri = document['uri']
            if http_verb == 'POST' and http_uri == '/sms':
                if 'msisdn' not in document['json']:
                    ch.basic_ack(delivery_tag=method.delivery_tag)
                elif ',' in document['json']['msisdn']:
                    # si on a plusieurs destinataires pour le mesage
                    # on remet dans la file d'attente des commandes
                    # pour chaque destinataire
                    msisdns = document['json']['msisdn'].replace(' ', '')
                    msisdns = msisdns.split(',')
                    for index, msisdn in enumerate(msisdns):
                        bar = {
                            'message': document['json'].get(
                                'message',
                                'message vide'
                                ),
                            'msisdn': msisdn,
                            'sender_id': document['json'].get(
                                'sender_id',
                                'default'
                                ),
                            'encoding': document['json'].get(
                                'encoding',
                                'DEFAULT'
                                )
                            }
                        data = {
                            key: value for key, value in document.items()
                            if key != 'json'
                            }
                        data['json'] = bar
                        data['index'] = index
                        ch.basic_publish(
                            exchange='sendsms',
                            routing_key=queue,
                            body=json.dumps(data),
                            properties=pika.BasicProperties(
                                delivery_mode=2
                                )
                            )
                    ch.basic_ack(delivery_tag=method.delivery_tag)
                else:
                    # on envoie sur la file d'attente la commande
                    # d'envoi de SMS
                    response = {
                        'code': '{}-2000'.format(SERVICE_CODE),
                        'status': 200,
                        'message': 'OK'
                        }
                    msisdn = document['json']['msisdn']
                    message = document['json'].get(
                        'message',
                        'message vide'
                        )
                    sender_id = document['json'].get(
                        'sender_id',
                        'default'
                        )
                    request_id = document['headers']['X-Request-Id']
                    log_msg = (
                        "POST simbank: tentative d'envoi "
                        "de sms vers {}"
                        ).format(msisdn)
                    journal.send(
                        log_msg,
                        REQUEST_ID=request_id
                        )
                    key = 'SIMBANK_SLOTS_{}'.format(sender_id)
                    source = random.choice([i for i in CONFIG[key]])
                    sms = Sms(source, msisdn, message)
                    try:
                        sms.send(document['headers'])
                    except Exception as exc:
                        log_msg = (
                            "Erreur survenue lors de l'envoi: {}"
                            ).format(exc)
                        journal.send(
                            log_msg,
                            REQUEST_ID=request_id
                            )
                        ch.basic_ack(delivery_tag=method.delivery_tag)
                        response['status'] = 500
                        response['code'] = SERVICE_CODE+'-5000'
                        response['error'] = 'Erreur interne'
                        response['message'] = {
                            'msisdn': msisdn,
                            'date': datetime.datetime.now().strftime(
                                '%d-%m-%Y %H:%M:%S'
                                ),
                            'error': exc
                            }
                    else:
                        log_msg = (
                            "POST simbank: delivre au simbank "
                            "pour envoi vers {}"
                            ).format(msisdn)
                        journal.send(
                            log_msg,
                            REQUEST_ID=request_id
                            )
                        ch.basic_ack(delivery_tag=method.delivery_tag)
                        response['info'] = 'Message envoyé'
                    infos = {i: j for i, j in response.items()}
                    infos['service'] = 'simbank'
                    infos['request_id'] = request_id
                    if 'index' in document:
                        infos['index'] = document['index']
                    requests.post(
                        'https://{fqdn}/requests'.format(fqdn=APIGW),
                        json=infos
                        )
                    time.sleep(0.25)
    connection = pika.BlockingConnection(
        pika.ConnectionParameters(
            host='localhost',
            virtual_host='/simbank'
            )
        )
    channel = connection.channel()
    channel.basic_qos(prefetch_count=1)
    channel.basic_consume(callback, queue=queue)
    channel.start_consuming()


class Worker(threading.Thread):

    def __init__(self, name, queue):
        threading.Thread.__init__(self)
        self.name = name
        self.queue = queue

    def run(self):
        process(self.queue)

# EOF
