#!/usr/bin/env python
# encoding: utf-8

import datetime
import requests
import logging
from .consts import CONFIG, LOGS, ORANGE_MONEY, REDIS_DB, IzyTV


class OrangeMoney(object):

    def __init__(self):
        self.url_api = ORANGE_MONEY["url_api"]

    def requestToken(self):
        LOGS.logger.info("\n\nOrangeMoney - requesting token...\n\n")
        try:
            response = requests.post(
                ORANGE_MONEY["url_token"],
                headers={
                    'Authorization': 'Basic {}'.format(
                        CONFIG["orangemoney"]["hashcode"]
                    )
                },
                data={'grant_type': 'client_credentials'}
            )
            LOGS.logger.info(response)
            if response.status_code == 200:
                LOGS.logger.info("\n\nOrangemoney - token request successful\n\n")
                return response.json()['access_token']
            return None
        except Exception as e:
            LOGS.logger.info('\n\nAn exception occured. Token request failed.\n')
            LOGS.logger.info(e)
            return None

    def paymentRequest(self, data):
        token = self.requestToken()
        LOGS.logger.info("\n\ntoken: {}\n\n".format(token))
        LOGS.logger.info("\n\n====== OrangeMoney paymentRequest =======\n\n")
        if token:
            or_data = {
                "order_id": data["operation_id"],
                "amount": data["amount"],
                "return_url": IzyTV["return_url"],
                "cancel_url": IzyTV["cancel_url"],
                "notif_url": IzyTV["notif_url"],
                "lang": "fr",
                "reference": "MerchantWP00014",
                "merchant_key": ORANGE_MONEY['merchant_key'],
                "currency": ORANGE_MONEY['currency'],
            }
            try:
                LOGS.logger.info("\n\nSending orangemoney payment request...\n\n")
                response = requests.post(
                    "{}/webpayment".format(ORANGE_MONEY["url_api"]),
                    headers={
                        'content-type': 'application/json',
                        'authorization': 'Bearer {}'.format(token)
                    },
                    json=or_data
                )
                LOGS.logger.info("\n\nStatus code: {}\n\n".format(response.status_code))
                LOGS.logger.info(response.json())

            except Exception as e:
                response = {
                    "code": "046-06-404",
                    "response": {},
                    "message": "Serveur Orange Money introuvable.",
                    "status": 404
                }
                LOGS.logger.error(str(e))

            if response.status_code == 201:
                LOGS.logger.info("\n\nOrangemoney payment request: SUCCESS.\n\n")
                LOGS.logger.info("\n\nStatus code: {}\n\n".format(response.status_code))
                jresponse = response.json()
                if jresponse.get("status"):
                    if jresponse["status"] == 201 and jresponse["message"] == "OK":
                        today = datetime.datetime.today()
                        transaction_date = today.strftime("%Y-%m-%d %H:%M:%S")
                        re_data = {
                            "TransactionDate": transaction_date,
                            "amount": data["amount"],
                            "order_id": data["operation_id"],
                            "label": data["label"],
                            "device_id": data["device_id"],
                            "other": data["other"],
                            "client_refnum": data["client_refnum"],
                            "offre_refnum": data["offre_refnum"],
                            "partner_ref": jresponse["pay_token"],
                            "operator": "orange-izytv",
                            "caller_num": "320000000",
                            "service_type": "izytv",
                            "status": "En attente de paiement",
                            "info": "Activation en attente",
                        }
                        LOGS.logger.info(re_data)
                        redis_key = "request:payment:{}".format(jresponse["pay_token"])
                        REDIS_DB.hmset(redis_key, re_data)
                        response = {
                            "code": "046-06-200",
                            "response": {
                                "message": jresponse["message"],
                                "pay_token": jresponse["pay_token"],
                                "notif_token": jresponse["notif_token"],
                                "payment_url": jresponse["payment_url"]
                            },
                            "message": jresponse["message"],
                            "status": 200
                        }
                    else:
                        response = {
                            "code": "046-06-{}".format(jresponse["status"]),
                            "response": {
                                "message": jresponse["message"],
                                "pay_token": jresponse["pay_token"],
                                "notif_token": jresponse["notif_token"],
                                "payment_url": jresponse["payment_url"]
                            },
                            "message": jresponse["message"],
                            "status": jresponse["status"]
                        }
                elif jresponse.get("code"):
                    response = {
                        "code": "046-06-{}".format(jresponse["code"]),
                        "response": {
                            "message": jresponse["description"],
                            "pay_token": None,
                            "notif_token": None,
                            "payment_url": None
                        },
                        "message": jresponse["message"],
                        "status": jresponse["code"]
                    }
            else:
                LOGS.logger.info("\n\nOrangemoney payment request: FAIL.\n\n")
                LOGS.logger.info("\n\nStatus code: {}\n\n".format(response.status_code))
                response = {
                    "code": "046-06-{}".format(response.status_code),
                    "response": {},
                    "message": "Serveur Orange Money introuvable",
                    "status": response.status_code
                }
                LOGS.logger.info("\n\nRequest unsuccesful\n\n")
        return response

    def checkTransactionStatus(self, data):
        auth_token = self.requestToken()
        LOGS.logger.info("\n\n====== OrangeMoney TransactionStatusRequest =======\n\n")
        LOGS.logger.info("\n\nauth_token: {}\n\n".format(auth_token))
        if auth_token:
            data = {
                "order_id": data["order_id"],
                "amount": data["amount"],
                "pay_token": data["pay_token"],
            }
            try:
                response = requests.post(
                    "{}/transactionstatus".format(ORANGE_MONEY["url_api"]),
                    headers={
                        'content-type': 'application/json',
                        'authorization': 'Bearer {}'.format(auth_token)
                    },
                    json=data
                )
                LOGS.logger.info(response.json())
            except Exception as e:
                LOGS.logger.warning("\n\nAn exception occured \n\n")
                LOGS.logger.error(e)
                response = {
                    "code": "046-06-404",
                    "response": {},
                    "message": "Serveur Orange Money introuvable",
                    "status": 404
                }
                return response
            if response.status_code == 201:
                jresponse = response.json()
                status = jresponse['status']
                choices = {
                    'SUCCESS': [200, 4],
                    'INITIATED': [000, 1],
                    'PENDING': [000, 2],
                    'EXPIRED': [000, 3],
                    'FAILED': [000, 3],
                }
                response = {
                    "code": "046-06-{:03d}".format(choices[status][0]),
                    "response": {
                        "ResponseCode": jresponse['status'],
                        "order_id": jresponse['order_id'],
                        "txnid": jresponse['txnid'],
                    },
                    "message": jresponse['status'],
                    "status": choices[status][-1],
                }
                LOGS.logger.info("\n\nFinal result\n\n")
                LOGS.logger.info(response)
            return response

    def checkTransactionStatusRedis(self, order_id):
        """Used by 4D in order to check a given transaction current status"""

        redis_key = "request:payment:{}".format(order_id)
        val = REDIS_DB.hgetall(redis_key.encode('utf-8'))
        data = {key.decode('utf-8'): value.decode('utf-8') for (key, value) in val.items()}
        response = {
            "code": 200,
            "response": {
                "order_id": data["partner_ref"]
            },
            "message": data["info"],
            "status": data["status"]
        }
        return response

# EOF
