#!/usr/bin/env python
# encoding: utf-8


from flask import jsonify, request

from blueflask.lib.errors import not_found
from blueflask.lib.infos import no_content

from ...objects import Department
from ...consts import SERVICE_CODE
from . import api


@api.route("/departments", methods=["GET"])
def departments():
    """
    Liste des départements de l'entrperise
    Renvoie une liste des départements.
    ---
    tags:
      - Departments
    parameters:
      - name: name
        in: query
        type: string
        description: Rechercher par nom
    responses:
      500:
        description: Une erreur interne à l'API s'est produite
      204:
        description: Aucun résultat n'a été trouvé
      200:
        description: Liste des départements
        schema:
          id: Departments list
          properties:
            length:
              type: integer
              description: Nombre de départements
            departments:
              type: array
              items:
                properties:
                  name:
                    type: string
                    description: Nom du département
    """
    name = request.args.get("name", None)
    if not name:
        departments = Department.all()
    else:
        departments = Department.search(name)
    if not departments:
        return no_content(service_code=SERVICE_CODE)
    else:
        result = {
            "length": len(departments),
            "departments": [{"name": department.name} for department in departments],
        }
        return jsonify(result)


@api.route("/departments/<string:name>", methods=["GET"])
def department(name):
    """
    Informations sur un département.
    Retourne toutes les informations d'un département inscrit dans l'annuaire
    ---
    tags:
      - Departments
    definitions:
      - schema:
          id: Error
          properties:
            error:
              type: string
              description: message court décrivant l'erreur rencontrée
            status:
              type: string
              description: statut de la requête HTTP
            code:
              type: integer
              description: code d'erreur
            message:
              type: string
              description: détails sur l'erreur
      - schema:
          id: Department
          properties:
            name:
              type: string
              description: Nom du département
            dn:
              type: string
              description: Adresse du département dans l'annuaire
            roles:
              type: array
              items:
                properties:
                  name:
                    type: string
                    description: Nom du rôle
                  dn:
                    type: string
                    description: Adresse du rôle dans l'annuaire
                  occupants:
                    type: array
                    items:
                      properties:
                        name:
                          type: string
                          description: Nom de l'utilisateur
                        username:
                          type: string
                          description: Nom d'utilisateur LDAP
    parameters:
      - name: name
        in: path
        type: string
        required: true
        description: nom du département
    responses:
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
      404:
        description: Aucun utilisateur n'a été trouvé
        $ref: "#/definitions/Error"
      401:
        description: Erreur d'authentification
        $ref: "#/definitions/Error"
      403:
        description: Action non autorisée
        $ref: "#/definitions/Error"
      204:
        description: Aucun résultat à retourner
      200:
        description: Informations sur le département
        $ref: "#/definitions/Department"
    """
    try:
        department = Department(name)
    except IndexError:
        msg = "{} n'existe pas".format(name)
        return not_found(msg, service_code=SERVICE_CODE)
    else:
        infos = {
            key: value
            for key, value in department.__dict__.items()
            if key not in ["handler", "ou"]
        }
        infos["roles"] = [
            {
                "name": role.name,
                "dn": role.dn,
                "occupants": [
                    {"name": occupant.name, "uid": occupant.uid}
                    for occupant in role.occupants
                ],
            }
            for role in department.roles
        ]
        return jsonify(infos)


# EOF
