#!/usr/bin/env python
# encoding: utf-8


from itsdangerous import TimedJSONWebSignatureSerializer as Serializer
from itsdangerous import JSONWebSignatureSerializer as JWS
from itsdangerous import BadSignature, SignatureExpired

from . import consts, core
from .consts import LOGS

import random
import string

import ldap

import sys

sys.getrecursionlimit(10000)

__all__ = ["Position", "Department", "SystemAccount", "User", "App", "AppGroup"]


class Position(core.LdapObject):

    """
    Cet objet représente un poste dans un département de l'entreprise.

    C'est un objet de type 'organizationalRole'.

    Ce genre d'objet est déterminé par son 'cn'
    """

    def __init__(self, name):

        super(Position, self).__init__("Position", name)

    @staticmethod
    def create(department_name, position_name):
        try:
            department = Department(department_name)
        except IndexError:
            raise ValueError("Le département '{}' n'existe pas".format(department_name))
        else:
            department.add_role(position_name)

    @property
    def occupants(self):

        result = []
        if hasattr(self, "roleOccupant"):
            try:
                if not isinstance(self.roleOccupant, list):
                    result = [User(self.roleOccupant)]
                else:
                    result = [User(dn) for dn in self.roleOccupant]
            except Exception:  # ldap.NO_SUCH_OBJECT
                pass
        return result

    @property
    def department(self):
        return Department(",".join(self.dn.split(",")[1:]))

    def add_occupant(self, user):
        """Ajouter un utilisateur occupant ce poste"""
        data = [(ldap.MOD_ADD, "roleOccupant", user.dn.encode("utf-8"))]
        core.ldap_conn("modify_s", self.dn, data)

    def remove_occupant(self, user):
        """Retirer un utilisateur de la liste des personnes à ce poste"""
        data = [(ldap.MOD_DELETE, "roleOccupant", user.dn.encode("utf-8"))]
        core.ldap_conn("modify_s", self.dn, data)


class Department(core.LdapObject):

    """
    Représente un département dans l'entreprise.

    C'est un objet de type 'organizationalUnit'.

    Ce genre d'objet est déterminé par son 'ou'
    """

    def __init__(self, name):

        super(Department, self).__init__("Department", name)

    @property
    def roles(self):
        """Retourne la liste des postes existant dans un département"""
        employeetypes = Position.all()
        result = [i for i in employeetypes if i.department.name == self.name]
        return result

    def add_role(self, name):
        """Créer un rôle dans ce département"""
        dn = "cn={},{}".format(name, self.dn)
        core.ldap_conn("add_s", dn, [("objectClass", b"organizationalRole")])


class App(core.LdapObject):
    def __init__(self, name):
        super(App, self).__init__("App", name)

    @property
    def groups(self):
        return [i for i in AppGroup.all() if i.app.name == self.name]


class AppGroup(core.LdapObject):
    def __init__(self, name):
        super(AppGroup, self).__init__("AppGroup", name)

    @property
    def members(self):
        result = []
        if hasattr(self, "memberUid"):
            try:
                if not isinstance(self.memberUid, list):
                    result = [User(self.memberUid)]
                else:
                    result = [User(i) for i in self.memberUid]
            except Exception:  # ldap.NO_SUCH_OBJECT
                pass
        return result

    @property
    def app(self):

        return App(",".join(self.dn.split(",")[1:]))


class User(core.LdapObject):

    """
    Représente un utilisateur humain.

    Il s'agit soit d'un employé, soit d'un partenaire.
    C'est un objet de type 'inetOrgPerson'.

    Ce genre d'objet est déterminé par son 'uid'
    """

    def __init__(self, name):

        super(User, self).__init__("User", name)

        if hasattr(self, "manager"):
            self.manager = User(self.manager)

    @property
    def department(self):
        """Avec l'attribut 'employeeType' on peut retrouver le departement"""

        return Position(self.employeeType).department

    @staticmethod
    def create(
        name, email, department, registration_number, role, mobile, manager, **kwargs
    ):

        users = User.all()
        for user in users:
            flag = user.cn == name or user.uid == email
            if flag:
                raise ValueError(
                    "Un utilisateur avec une de ces informations existe déjà"
                )
        else:
            # on vérifie si le département existe
            try:
                department = Department(department)
            except IndexError:
                raise ValueError(
                    "Le département '{}' n'existe pas".format(department.name)
                )
            else:
                employeeType = [
                    employeetype
                    for employeetype in department.roles
                    if employeetype.name == role
                ]
                if employeeType and len(employeeType) > 1:
                    raise ValueError("Le rôle '{}' n'est pas unique".format(role))
                elif len(employeeType) == 1:
                    employeeType = employeeType[0]
                else:
                    department.add_role(role)
                    employeeType = Position(role)
                if len(name.split()) == 2:
                    first_name, last_name = name.split()
                elif len(name.split()) > 2:
                    last_name = name.split()[-1]
                    first_name = " ".join(name.split()[:-1])
                elif len(name.split()) == 1:
                    last_name = name.split()[0]
                    first_name = ""
                first_name = kwargs.get("givenName", first_name)
                last_name = kwargs.get("sn", last_name)
                mobile = kwargs.get("mobile", mobile)
                manager = kwargs.get("manager", manager)
            data = {
                "uid": email,
                "mail": email,
                "givenName": first_name,
                "sn": last_name,
                "cn": name,
                "employeeType": employeeType.name,
                "employeeNumber": registration_number,
                "objectClass": "inetOrgPerson",
            }
            if mobile:
                data["mobile"] = mobile
            if manager:
                data["manager"] = manager
            print(data)
            data = [(key, value.encode("utf-8")) for key, value in data.items()]
            domain = consts.CONFIG["User"]["dn"]
            subdomain = kwargs.get("subdomain")
            if subdomain:
                domain = "ou={},{}".format(subdomain, domain)
            LOGS.logger.info(data)
            core.ldap_conn("add_s", "cn={0},{1}".format(name, domain), data)
            # utilisateur créé
            user = User(email)
            # maintenant on le rajoute dans le groupe des occupants du rôle
            employeeType.add_occupant(user)
            # on modifie aussi son uidNumber et son mot de passe
            uidNumbers = [
                int(user.uidNumber) for user in users if hasattr(user, "uidNumber")
            ]
            uidnumber = max(uidNumbers) + 1
            uidnumber = str(uidnumber)
            data = {
                "userPassword": email,
                "gidNumber": "501",
                "uidNumber": uidnumber,
                "homeDirectory": "/home/{}".format(email.split("@")[0]),
                "objectClass": "posixAccount",
            }
            user.update(replace=False, **data)
            user.reset()

    @staticmethod
    def generate_random_password():
        """Création d'un mot de passe au hasard"""
        choices = string.ascii_letters + string.digits
        new_password = "".join(random.sample(choices, random.randint(8, 16)))
        return new_password

    @staticmethod
    def verify_auth_token(token):
        """Vérification de la validité d'un jeton de session"""
        secret_key = list(consts.CONFIG["SECRET_KEY"].keys())[0]
        s = Serializer(secret_key)
        try:
            data = s.loads(token)
        except (BadSignature, SignatureExpired, TypeError):
            return None
        else:
            return User(data["uid"])

    def generate_auth_token(self):
        """Création d'un jeton de session valide une heure"""
        secret_key = list(consts.CONFIG["SECRET_KEY"].keys())[0]
        s = Serializer(secret_key, expires_in=3600)
        data = {
            i: j
            for i, j in self.__dict__.items()
            if i not in ["uid", "userPassword", "manager", "handler"]
        }
        data["uid"] = self.uid[0]
        return s.dumps(data).decode("utf-8")

    def authenticate(self, password):
        """Vérification du mot de passe de l'utilisateur"""
        c = ldap.initialize(consts.LDAP_SERVER)
        try:
            c.bind_s(self.dn, password)
        except ldap.INVALID_CREDENTIALS:
            return False
        else:
            c.unbind_s()
            return True

    def passwd(self, old_password, new_password):
        """Modification du mot de passe"""
        core.ldap_conn("passwd_s", self.dn, old_password, new_password)

    def reset(self):
        """Reset du mot de passe"""
        bar = self.uid[0]
        self.update(userPassword=bar)
        new_password = User.generate_random_password()
        self.passwd(bar, new_password)
        core.Email().reset(self, new_password)
        return new_password

    def update(self, replace=True, **kwargs):
        """Mise à jour d'un ou plusieurs paramètres"""
        if replace:
            action = ldap.MOD_REPLACE
        else:
            action = ldap.MOD_ADD
        data = [(action, key, value.encode("utf-8")) for key, value in kwargs.items()]
        core.ldap_conn("modify_s", self.dn, data)

    def delete(self):
        Position(self.employeeType).remove_occupant(self)
        core.ldap_conn("delete_s", self.dn)

    def apps_roles(self):
        """Retourne la liste des rôles spécifiques
        que l'utilisateur possède dans diverses applications"""
        roles = [
            role
            for role in AppGroup.all()
            if self.name in [occupant.name for occupant in role.members]
        ]
        return roles

    def apps(self):
        """Retourne la liste des applications
        où l'utilisateur a un rôle spécifique"""
        apps = set()
        for role in self.apps_roles():
            apps.add(role.app)
        return list(apps)

    def is_blueldap_admin(self):
        for role in self.apps_roles:
            if role.app.name == "blueldap" and role.name == "blueldap_admins":
                return True
        else:
            return False


class SystemAccount(core.LdapObject):

    """
    Représente un compte système.

    Il s'agit d'un compte générique utilisé par des systèmes
    plutôt que par des personnes physiques.
    Ce sont notamment les comptes utilisés par des programmes,
    des applications, etc.
    C'est un objet de type 'simpleSecurityObject' et en même temps 'account'.

    Ce genre d'objet est déterminé par son 'uid'
    """

    def __init__(self, name):

        super(SystemAccount, self).__init__("SystemAccount", name)

    @staticmethod
    def create(username, host, email):
        accounts = [
            SystemAccount(i) for i, j in core.LdapInspector("SystemAccount").search()
        ]
        for account in accounts:
            if account.uid == username:
                raise ValueError("Un compte avec les mêmes informations existe déjà")
        else:
            data = {
                "uid": username,
                "objectClass": "simpleSecurityObject",
                "userPassword": username,
                "host": host,
            }
            data = [(key, value.encode("utf-8")) for key, value in data.items()]
            domain = consts.CONFIG["SystemAccount"]["dn"]
            core.ldap_conn("add_s", domain, data)
            account = SystemAccount(username)
            account.reset()

    @staticmethod
    def generate_random_password():
        """Création d'un mot de passe au hasard"""
        choices = string.ascii_letters + string.digits
        new_password = "".join(random.sample(choices, random.randint(8, 16)))
        return new_password

    @staticmethod
    def verify_auth_token(token):
        """Vérification de la validité d'un jeton de session"""
        secret_key = list(consts.CONFIG["SECRET_KEY"].keys())[0]
        s = Serializer(secret_key)
        try:
            data = s.loads(token)
        except (BadSignature, SignatureExpired, TypeError):
            return None
        else:
            return SystemAccount(data["uid"])

    def generate_auth_token(self, seconds=3600):
        """Création d'un jeton de session valide une heure par défaut"""
        secret_key = list(consts.CONFIG["SECRET_KEY"].keys())[0]
        if not seconds:
            s = JWS(secret_key)
        else:
            s = Serializer(secret_key, expires_in=seconds)
        data = {
            i: j
            for i, j in self.__dict__.items()
            if i not in ["uid", "userPassword", "handler"]
        }
        data["uid"] = self.uid[0]
        return s.dumps(data).decode("utf-8")

    def authenticate(self, password):
        """Vérification du mot de passe de l'utilisateur"""
        c = ldap.initialize(consts.LDAP_SERVER)
        try:
            c.bind_s(self.dn, password)
        except ldap.INVALID_CREDENTIALS:
            return False
        else:
            c.unbind_s()
            return True

    def passwd(self, old_password, new_password):
        """Modification du mot de passe"""
        core.ldap_conn("passwd_s", self.dn, old_password, new_password)

    def reset(self):
        """Reset du mot de passe"""
        self.update(userPassword=self.uid)
        new_password = SystemAccount.generate_random_password()
        self.passwd(self.uid, new_password)
        return new_password

    def is_blueldap_admin(self):
        if self.uid == "etech":
            return True
        else:
            return False


# EOF
