#!/usr/bin/env python
# encoding: utf-8


"""
App to interact with an OpenLDAP instance.
Requires: python3, pyldap, click
"""


from email.mime.text import MIMEText

from . import consts

import base64
import ldap
import smtplib


__all__ = [
    'Email',
    'LdapInspector',
    'LdapObject'
    ]


def ldap_conn(command, *args, **kwargs):
    conn = consts.LDAP_CONN
    conn.bind(
        consts.LDAP_ADMIN_DN,
        consts.CONFIG['LDAP']['LDAP_ADMIN_PASSWORD']
        )
    result = getattr(conn, command)(*args, **kwargs)
    return result


class Email(object):
    """Notifications"""

    def __init__(self):
        self.From = 'support_n1@si.blueline.mg'
        self.smtp = smtplib.SMTP('localhost')

    def send(self, to):
        self.text += (
            'Vous pouvez changer votre mot de passe '
            'à l\'adresse suivante: {address}\n\n'
            'Pour plus d\'informations, merci de contacter '
            'support_n1@si.blueline.mg\n\n'
            'Cordialement\n--\nLDAP Blueline'
            ).format(
                address=consts.CONFIG.get('LDAP', 'RESET_LINK')
                )
        msg = MIMEText(self.text)
        msg['Subject'] = "Votre compte LDAP Blueline"
        msg['From'] = self.From
        msg['To'] = to
        self.smtp.send_message(msg)

    def create(self, user, password):
        """email sent when account is created"""

        self.text = (
            'Bonjour {name}.\n'
            'Votre compte LDAP blueline a été créé.\n\n'
            'Login: {username}\n'
            'Mot de passe: {password}\n\n'
            'Ce compte LDAP représente votre identité dans '
            'l\'annuaire interne de l\'entreprise.\n\n'
            ).format(
                name=user.cn,
                username=user.username,
                dn=user.dn,
                password=password
                )
        self.send(user.mail)

    def reset(self, user, password):
        """email sent when an account's password is reset"""

        if isinstance(user.uid, list):
            username = user.uid[0]
        elif isinstance(user.uid, str):
            username = user.uid
        self.text = (
            'Bonjour {name}.\n'
            'Votre mot de passe LDAP blueline a été réinitialisé.\n\n'
            'Login: {username}\n'
            'Mot de passe: {password}\n\n'
            ).format(
                name=user.cn,
                username=username,
                password=password
                )
        if hasattr(user, 'mail'):
            self.send(user.mail)


class LdapInspector(object):

    """Encapsulation des méthodes de base pour gérer un type d'objet"""

    def __init__(self, objectType):
        if objectType not in consts.CONFIG:
            raise ValueError(
                "'{objectType}' n'existe pas dans la configuration".format(
                    objectType=objectType)
                )
        else:
            self.objectType = objectType
            obj = consts.CONFIG[self.objectType]
            self.top_domain = obj['dn']
            self.attributes = obj['attributes'].split(', ')
            self.identifier = obj['identifier']
            self.dn_begin = obj['dn_begin']
            self.object_class = obj['object_class']

    def __repr__(self):
        return '<LdapInspector {}>'.format(self.objectType)

    def read(self, dn):
        """Lecture des informations sous un domaine précis"""
        result = ldap_conn('read_s', dn, attrlist=self.attributes)
        bar = {}
        for key, value in result.items():
            _value = []
            for i in value:
                if key == 'jpegPhoto':
                    i = base64.b64encode(i)
                i = i.decode('utf-8')
                _value.append(i)
            value = [i for i in _value]
            if len(value) == 1:
                value = value[0]
            bar[key] = value
        return dn, bar

    def search(self, parameter=None, identifier=False):
        """Recherche d'un ou de tout objet de la même classe"""
        if identifier and not parameter:
            raise ValueError(
                (
                    "Impossible d'effectuer la recherche. "
                    "Ce que vous demandez est impossible."
                    )
                )
        filterstr = '(objectClass={object_class})'.format(
            object_class=self.object_class
            )
        if parameter and identifier:
            filterstr = (
                '(&'
                '{filterstr}'
                '({identifier}={parameter})'
                ')'
                ).format(
                    filterstr=filterstr,
                    identifier=self.identifier,
                    parameter=parameter
                    )
        data = ldap_conn(
            'search_s',
            self.top_domain,
            ldap.SCOPE_SUBTREE,
            filterstr=filterstr,
            attrlist=self.attributes
            )
        result = []
        for dn, infos in data:
            if dn == self.top_domain:
                continue
            bar = {}
            for key, value in infos.items():
                _value = []
                for i in value:
                    if key == 'jpegPhoto':
                        i = base64.b64encode(i)
                    i = i.decode('utf-8')
                    _value.append(i)
                value = [i for i in _value]
                if len(value) == 1:
                    value = value[0]
                bar[key] = value
            result.append((dn, bar))
        if parameter:
            result = [
                (dn, infos) for dn, infos in result
                if parameter.lower() in dn.lower() or
                parameter.lower() in infos.get('uid', '')
                ]
        return result


class LdapObject(object):

    def __init__(self, object_class, parameter):

        self.handler = LdapInspector(object_class)
        dn_begin = self.handler.dn_begin

        # suivant le paramètre on recherche l'objet dans l'arbre LDAP
        try:
            int(parameter)  # on vérifie si le paramètre n'est pas un matricule
        except (ValueError, TypeError):
            if parameter.startswith('{dn_begin}='.format(dn_begin=dn_begin)):
                try:
                    dn, infos = self.handler.read(parameter)
                except ldap.NO_SUCH_OBJECT:
                    raise IndexError("No user '{}'".format(parameter))
            else:
                dn, infos = self.handler.search(parameter, identifier=True)[0]
        else:
            if object_class == 'User':
                for dn, infos in self.handler.search():
                    if infos.get('employeeNumber') == parameter:
                        break
                else:
                    raise IndexError("No user '{}'".format(parameter))

        self.dn = dn
        self.name = self.dn.split(',')[0].split('=')[1]
        for key, value in infos.items():
            if key == 'uid' and isinstance(value, str):
                value = [value]
            setattr(self, key, value)

    def __repr__(self):
        return '<{} {}>'.format(self.__class__.__name__, self.name)

    @classmethod
    def all(cls):
        handler = LdapInspector(cls.__name__)
        result = handler.search()
        result = [cls(dn) for dn, infos in result]
        return result

    @classmethod
    def search(cls, parameter):
        handler = LdapInspector(cls.__name__)
        result = handler.search(parameter)
        result = [cls(dn) for dn, infos in result]
        return result

    def update(self, key, value, add=False):
        """Modification d'une information"""
        value = value.encode('utf-8')
        action = ldap.MOD_REPLACE
        if add:
            action = ldap.MOD_ADD
        ldap_conn('modify_s', self.dn, [(action, key, value)])

# EOF
