#!/usr/bin/env python
# encoding: utf-8

from math import radians, cos, sin, asin, sqrt
from datetime import datetime
from decimal import Decimal
from . import TZ  # , LOGS
import time


class Position(object):
    """représentation d'une position GPS"""
    def __init__(self, timestamp, longitude, latitude, angle, speed):
        self.datetime = datetime.fromtimestamp(timestamp, tz=TZ)
        self.longitude = longitude
        self.latitude = latitude
        self.angle = float(angle)
        self.speed = float(speed)

    def json(self):
        "javascript object notation"
        result = {
            'datetime': self.datetime.strftime('%Y-%m-%d %H:%M:%S %Z'),
            'coords': (self.longitude, self.latitude, self.angle, self.speed)
            }
        return result

    def haversine_from(self, position):
        """extrait de http://bit.ly/1LDKHF7: distance depuis un autre point"""
        # conversion en radians
        lon1, lat1, lon2, lat2 = map(
            radians,
            [
                position.longitude,
                position.latitude,
                self.longitude,
                self.latitude
                ]
            )

        # formule haversine
        dlon = lon2 - lon1
        dlat = lat2 - lat1
        a = sin(dlat/2)**2 + cos(lat1) * cos(lat2) * sin(dlon/2)**2
        c = 2 * asin(sqrt(a))

        # 6371 km c'est le rayon de la Terre
        km = 6371 * c
        return km

    @staticmethod
    def fromjson(document):
        """créer un objet Position à partir d'une chaine JSON"""
        document = eval(document)
        timestamp = time.mktime(
            time.strptime(document['datetime'], '%Y-%m-%d %H:%M:%S %Z')
            )
        return Position(timestamp, *document['coords'])

    def __repr__(self):
        return '<Position {}>'.format(self.json())

# EOF
