#!/usr/local/bin python
# encoding: utf-8

from blueflask.lib.errors import bad_request, internal_error
from flask import jsonify, request, url_for
from ...positions import Position
from ...devices import Device
from ...core import Track
from ...decorators import json
# from ... import LOGS
from . import api
import time


@api.route("/last_positions/<string:imei>", methods=['GET'])
def last_positions(imei):
    device = Device(imei)
    position = device.last_position
    if not position:
        response = {
                'status': 204,
                'message': 'correctly processed: no content to return'
            }
        return jsonify(response)
    elif not type(position) is Position:
        return internal_error(position)
    else:
        point = {
            'geometry': {
                'type': 'Point',
                'coordinates': [
                    float(position.longitude),
                    float(position.latitude)
                    ],
                },
            'type': 'Feature',
            'properties': {
                'datetime': time.mktime(position.datetime.timetuple()),
                'speed': float(position.speed)
                }
            }
        return jsonify(point)


@api.route("/history/<string:imei>", methods=["GET"])
def history(imei):
    """Télécharger un résumé du parcours des véhicules"""
    device = Device(imei)
    data = request.get_json()
    start = data.get('start', None)
    stop = data.get('stop', None)
    if not start:
        return bad_request("vous devez spécifier une date de début")
    else:
        # SI avec params start et stpo, on recupère la trace
        positions = device.history(start, stop)
        if not positions:
            response = {
                    'status': 204,
                    'message': 'correctly processed: no content to return'
                }
            return jsonify(response)
        else:
            result = {
                'type': 'FeatureCollection',
                'features': [],
                'totaldistance': []
                }
            linestring = {
                'geometry': {
                    'type': 'LineString',
                    'coordinates': []
                    },
                'type': 'Feature'
                }
            distance = 0
            for index, position in enumerate(positions):
                linestring['geometry']['coordinates'].append(
                    [
                        float(position.longitude),
                        float(position.latitude)
                        ]
                    )
                data = {
                    'type': 'Feature',
                    'properties': {
                        'speed': float(position.speed),
                        'angle': float(position.angle),
                        'date': position.datetime.strftime('%d-%m-%Y'),
                        'hour': position.datetime.strftime('%H:%M:%S'),
                        'distance': 'distance',
                        'label': 'step'
                        },
                    'geometry': {
                        'type': 'Point',
                        'coordinates': [
                            float(position.longitude),
                            float(position.latitude)
                            ]
                        }
                    }
                if index == 0:
                    data['properties']['label'] = 'start'
                    data['properties']['distance'] = 0
                else:
                    previous = positions[index-1]
                    time_interval = (
                        position.datetime - previous.datetime
                        ).total_seconds()
                    distance += position.haversine_from(previous)
                    data['properties']['distance'] = round(distance, 4)
                    if index == len(positions)-1:
                        data['properties']['label'] = 'stop'
                        result['totaldistance'].append(distance)
                    elif position.speed == 0 and time_interval > 300:
                        data['properties']['label'] = 'pause'

                result['features'].append(data)
            result['features'].append(linestring)

            return jsonify(result)


@api.route('/distances/<string:imei>', methods=['GET'])
@json
def distances(imei):
    """récupération de la distance parcourue par un véhicule"""
    now = time.localtime()
    data = request.get_json()
    pattern = '%Y-%m-%d %H:%M:%S'
    day = "{year}-{month}-{day}".format(
        year=now.tm_year, month=now.tm_mon, day=now.tm_mday
        )
    device = Device(imei)
    start = data.get('start', "{0} 00:00:00".format(day))
    stop = stop = data.get(
        'stop',
        "{day} {hour}:{minutes}:{seconds}".format(
            day=day,
            hour=str(now.tm_hour).zfill(2),
            minutes=str(now.tm_min).zfill(2),
            seconds=str(now.tm_sec).zfill(2)
            )
        )
    if time.strptime(start, pattern) > time.strptime(stop, pattern):
        msg = "start cannot be later than stop"
        return bad_request(msg)
    else:
        positions = device.history(start, stop)
        if not positions:
            response = jsonify(
                    {
                        'status': 204,
                        'message': 'correctly processed: no content to return'
                    }
                )
            response.status_code = 204
            return response
        else:
            distance = 0
            for index, position in enumerate(positions):
                if index > 0:
                    distance += position.haversine_from(positions[index-1])
            distance = round(distance, 4)
            return {'imei': imei, 'distance': distance}


@api.route("/positions/<string:imei>", methods=['GET', 'POST'])
def positions(imei):
    """récupération, enregistrement des traces d'un device"""
    device = Device(imei)
    if request.method == 'POST':
        position = Position(**dict(request.form.items()))
        track = Track(device, position)
        result = track.store()
        if type(result) is Track:
            response = jsonify(
                {
                    'status': 201,
                    'message': 'track stored'
                }
            )
            response.status_code = 201
            response.headers['Location'] = url_for('api.positions', imei=imei)
            return response
        else:
            return internal_error(result)
