#!/usr/bin/env python
# encoding: utf-8

"""
Ce module contient les briques de bases du service
de gestion des politiques de consommation
des utilisateurs Internet chez Blueline.

Une synchronisation des données est effectuée tous les matins
avec Aiguillier pour disposer d'une base locale afin de faire
des calculs sans craindre les ralentissements réseau. Toutes
les vérifications se font donc en local.
"""
import logging
import smtplib
import json
from email.mime.base import MIMEBase
from datetime import datetime as dt, timedelta

import requests

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate

from .consts import FUPDB_CREDENTIALS, APIGW, LOGS, CONFIG

import records
import psycopg2


__all__ = [
    'database_url',
    'record_customer',
    'apply_product_policies',
    'update_customer_cdr'
    ]


def connexion(**kwargs):
    con = psycopg2.connect(
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        database=kwargs.get('name')
    )
    return con


def database_url(**kwargs):
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=kwargs.get('engine'),
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        name=kwargs.get('name')
        )
    return url


def si_database_url():
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=CONFIG['DB']['engine'],
        user=CONFIG['DB']['user'],
        password=CONFIG['DB']['password'],
        host=CONFIG['DB']['host'],
        port=CONFIG['DB']['port'],
        name=CONFIG['DB']['name'],
    )
    return url

def bi_database_url():
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=CONFIG['BI']['engine'],
        user=CONFIG['BI']['user'],
        password=CONFIG['BI']['password'],
        host=CONFIG['BI']['host'],
        port=CONFIG['BI']['port'],
        name=CONFIG['BI']['table_name'],
    )
    return url


def get_fuped_aiguiller():
    """
    Fetches all fuped customers in Aiguiller.
    The called API returns both BI and SI customers and results need to be
    filtered accordingly.
    """
    try:
        response = requests.get(
            CONFIG['AIGUILLIER']['url'] + '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password'],
            )
        )
        return response.json()['fup']
    except Exception as e:
        operation = "get fuped customers in Aiguiller"
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def get_customers_si(identifier=None, value=None):
    """Fetches fup-able customers in fup-service database."""
    if identifier and value:
        sql = (
            "SELECT * FROM customer where {}='{}'"
            .format(identifier, value)
        )
    else:
        sql = "SELECT * FROM customer"
    try:
        with records.Database(si_database_url()) as db:
            rows = db.query(sql)
            customer_l = rows.as_dict()
            return customer_l
    except Exception as e:
        operation = "fetch customers list in SI database"
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def get_apigw_token():
    """
    Fetches authentication token for requests to send to API GATEWAY services.
    """
    try:
        token_response = requests.post(
            CONFIG['APIGATEWAY']['token_url'],
            auth=(
                CONFIG['APIGATEWAY']['username'],
                CONFIG['APIGATEWAY']['password']
            ),
        )
        LOGS.logger.info(
            "Get api gateway authentication token: {}"
            .format(token_response.status_code)
        )
        return token_response.json()['token']
    except Exception as e:
        operation = "get api gateway token"
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        raise e


def get_customers_crm(product_name):
    """
    Fetches all customers information from CRM(4D) with their products.
    Only business_access, business_plus and airfiber (aka airfiber_prime)
    will be returned by this API.
    :param product_name: ["business_access", "business_plus", "airfiber"]
    :return: list of customers owning the provided product and
    their information
    """
    token = get_apigw_token()
    try:
        response = requests.post(
            CONFIG['APIGATEWAY']['crm_customers_url'],
            headers={'Content-Type': 'application/json', },
            data=json.dumps({'type': product_name, 'version': '2'}),
            auth=(token, ''),
        )
        LOGS.logger.info(
            "Get 4D customer list for product '{}' : {}"
            .format(product_name, response.status_code)
        )
        return response.json()['root']['data']['list']['produit']
    except Exception as e:
        operation = "get 4D customer list"
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def get_history(customer_name, date):
    """
    Get a given customer's fup history in fup-services database.
    :param customer_name: name of the customer, already lowered
    :param date: str YYYY-mm-dd
    """
    try:
        with records.Database(si_database_url()) as db:
            sql = (
                "SELECT * FROM fup "
                "WHERE datetime='{date}'::date "
                "AND customer='{name}' "
            ).format(name=customer_name, date=date)
            rows = db.query(sql)
            fup_history = rows.as_dict()
        LOGS.logger.info(
            "Customer {} fup history for {}: {}"
            .format(customer_name, date, fup_history)
        )
        return fup_history[0]
    except Exception as e:
        operation = (
            "get customer {}'s fup history in SI database"
            .format(customer_name)
        )
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []

# =============================== customers ==================================


def sync_customer(name, product_name, refnum):
    """
    Synchronizes (create or update) the customer with its instance in database.
    """
    exists = get_customers_si(
        identifier="name",
        value=name
    )
    if exists:
        if exists[0]['refnum'] != refnum or exists[0]['product'] != product_name:
            update_customer(name, product_name, refnum)
    else:
        create_customer(name, product_name, refnum)


def update_customer(customer_name, product_name, primary_ref):
    """
    Updates a customer in fup-services database
    """
    sql = (
        "UPDATE customer SET refnum='{ref}', product='{product}'"
        "WHERE name='{name}';"
        .format(ref=primary_ref, name=customer_name, product=product_name)
    )
    try:
        with records.Database(si_database_url()) as db:
            db.query(sql)
        LOGS.logger.info(
            "Customer {}[{}] with product {} updated in SI database"
            .format(customer_name, primary_ref, product_name)
        )
    except Exception as e:
        operation = (
            "update customer {} in SI database"
            .format(customer_name)
        )
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def create_customer(customer_name, product_name, primary_ref):
    """
    Creates a new customer entry in fup-services database.
    """
    sql = (
        "INSERT INTO customer(refnum, name, product)"
        "VALUES('{ref}', '{name}', '{product}');"
        .format(ref=primary_ref, name=customer_name, product=product_name)
    )
    try:
        with records.Database(si_database_url()) as db:
            db.query(sql)
        LOGS.logger.info(
            "Customer {}[{}] with product {} created in SI database"
            .format(customer_name, primary_ref, product_name)
        )
    except Exception as e:
        operation = (
            "create customer {} in SI database"
            .format(customer_name)
        )
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def purge_customer(name):
    """
    Remove all information stored in database about the given customer
    """
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = "DELETE FROM customer WHERE name=:name"
        db.query(sql, name=name)
        LOGS.logger.info("Customer {} information deleted from SI database".format(name))
        sql = "DELETE FROM cdr WHERE customer=:name"
        db.query(sql, name=name)
        LOGS.logger.info("Customer {} CDRs deleted from SI database".format(name))
        sql = "DELETE FROM fup WHERE customer=:name"
        db.query(sql, name=name)
        LOGS.logger.info("Customer {} fup history deleted from SI database".format(name))
    LOGS.logger.info("Customer {} purged from SI database".format(name))


def clean_database():
    """
    Removes all data in database about customers that are not returned by CRM list
    """
    customers_crm = []
    for product in ["business_access", "business_plus"]:
        customers = get_customers_crm(product)
        customers_crm += [customer['ident'].lower() for customer in customers]

    customers_si = get_customers_si()
    customers_si = [customer['name'] for customer in customers_si]

    customers_common = set(customers_crm) & set(customers_si)
    remainder_si = [name for name in customers_si if name not in customers_common]
    for customer in remainder_si:
        purge_customer(customer)
    LOGS.logger.info("{} customers were purged from database".format(len(remainder_si)))


def fup_customer(customer_name):
    """Fup the customer in Aiguiller (hp only). Operation is attempted 3 times."""
    fuped = False
    for attempt in range(1, 3):
        try:
            response = requests.post(
                CONFIG['AIGUILLIER']['url'] + "/fup?customer={}&period=hp".format(customer_name),
                auth=(
                    CONFIG['AIGUILLIER']['username'],
                    CONFIG['AIGUILLIER']['password']
                )
            )
            if response.status_code == 200:
                fuped = True
                break
            else:
                raise ValueError(response.text)
        except Exception as e:
            LOGS.logger.warning(
                "[FUP] Try {} out of 3. "
                "Unable to add customer {} to DT database: {} "
                .format(attempt, customer_name, str(e))
            )
    if fuped:
        LOGS.logger.info(
            "[FUP] Customer {} added to DT database"
            .format(customer_name)
        )
    else:
        LOGS.logger.error(
            "[FUP] Unable to add customer {} to DT database"
            .format(customer_name)
        )
        send_error_report(
            "FUP customer {} from Aiguillier".format(customer_name),
            "cf. /var/log/fup-services/fup-services.log"
        )


def defup_customer(customer_name):
    """
    Removes customer from DT database via Aiguiller.
    Operation is attempted 3 times.
    """
    defuped = False
    for attempt in range(1, 3):
        try:
            response = requests.delete(
                CONFIG['AIGUILLIER']['url_v2'] + '/consommation?remettre_normal={}&periode=hp'
                .format(customer_name),
                auth=(
                    CONFIG['AIGUILLIER']['username'],
                    CONFIG['AIGUILLIER']['password']
                )
            )
            if response.status_code == 200 or response.status_code == 409:
                defuped = True
                break
            else:
                raise ValueError(response.text)
        except Exception as e:
            LOGS.logger.warning(
                "[DEFUP] Try {} out of 3. "
                "Unable to remove customer {} from DT database: {} "
                .format(attempt, customer_name, str(e))
            )
    if defuped:
        LOGS.logger.info(
            "[DEFUP] Customer {} removed from DT database"
            .format(customer_name)
        )
    else:
        LOGS.logger.error(
            "[DEFUP] Unable to remove customer {} from DT database"
            .format(customer_name)
        )
        send_error_report(
            "DEFUP customer {} from Aiguillier".format(customer_name),
            "cf. /var/log/fup-services/fup-services.log"
        )


def record_customer(infos):
    """
    Add the given customer and its information to database
    """
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = (
            "INSERT INTO customer (name, refnum, product) "
            "VALUES (:name, :refnum, :product)"
            )
        db.query(
            sql,
            name=infos['name'],
            refnum=infos['refnum'],
            product=infos['product']
            )


# ================================ Products ==================================

def get_product_policy(product):
    """
    Fetches the policy information related to the product.
    :param product: the product name
    :return : {
        'max_data': maximum consumption allowed,
        'start_time': consumption starting time (HHmmss),
        'end_time': consumption ending time (HHmmss),
        'start_date': consumption starting day (isodow),
        'end_date': consumption ending day (isodow),
    }
    """
    policy_raw = CONFIG['POLICIES'][product].split(',')
    policy = {
        'max_data': policy_raw[0],
        'start_time':  policy_raw[1],
        'end_time': policy_raw[2],
        'start_date': policy_raw[3],
        'ending_date': policy_raw[4]
    }
    return policy


# =============================== CDRs =====================================

def update_consommation(product, policy, customer_name, fup_status, consumption):
    yesterday = (dt.today() - timedelta(days=1)).strftime("%Y-%m-%d")
    sql = (
        "INSERT INTO fup(product, policy, customer, datetime, status, conso)"
        "VALUES ('{product}', '{policy}', '{customer}', '{date}', '{status}','{consumption}');"
    ).format(
        product=product,
        policy=policy,
        customer=customer_name,
        date=yesterday,
        status=fup_status,
        consumption=consumption
    )
    try:
        with records.Database(si_database_url()) as db:
            db.query(sql)
        LOGS.logger.info("Customer {} consumption history updated.".format(customer_name))
    except Exception as e:
        operation = (
            "update customer {} consumption history"
            .format(customer_name)
        )
        LOGS.logger.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def get_consommation_sum(customer, date):
    """
    Fetches the last 30days consumptions of a customer from BI database
    """
    sql = (
        "SELECT SUM(octetsin+octetsout) "
        "FROM all_cdr "
        "WHERE clientname ilike '%{customer}%' "
        "AND sessiontime >= '{date}'::date "  # YYYY-MM-DD
        "AND source IN ('internet','cache') "
        "AND (extract('ISODOW' FROM sessiontime) between 1 AND 5) "
        "AND extract(HOUR FROM sessiontime) < 19 "
        "AND extract(HOUR FROM sessiontime) > 8 "
        "GROUP BY clientname;"
    ).format(customer=customer, date=date)
    try:
        with records.Database(bi_database_url()) as db:
            rows = db.query(sql)
            total = rows.as_dict()
            if total:
                total = float(total[0].get('sum', 0))
            else:
                total = 0
            LOGS.logger.info(
                "Customer {} 30days total consumption: {} ({}Go)"
                .format(customer, total, total/1000000000)
            )
            return total
    except Exception as e:
        operation = "fetch customer {} total consumption for the last 30days".format(customer)
        LOGS.logger.error("Unable to {}: {}".format(operation, str(e)))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return 0


def apply_product_policies(for_real, customer):
    """Sous routine appliquant les politiques de bon usage
    à un client en particulier.
    Utile pour ensuite faire de l'exécution parallèle"""
    response = customer.apply_product_policies(for_real=for_real)
    data = [
        policy for policy in response
        # if 'fup_in' in policy or 'fup_out' in policy
        ]
    return customer, data


def update_customer_cdr(last_date, customer):
    """Mise à jour des CDRs d'un client dans la base REDIS"""
    customer.get_cdr_from_aiguillier(
        last_date=last_date
        )

# ============================== utils =======================================

def send_error_report(operation, error):
    """
    Sends an email to development and monitoring team to notify an error
    in the process.
    """
    message = (
        "Service: {} [{}]\n\n"
        "Affected operation: {}\n\n"
        "Error: {}"
        .format("fup-services", APIGW, operation, error)
    )
    sender = "api-gateway@si.blueline.mg"
    # destinations = ["dev@si.blueline.mg", "sysadmin@si.blueline.mg"]
    destinations = ["yona.rasolonjatovo@staff.blueline.mg",]
    content = MIMEText(message, 'plain', 'utf-8')

    msg = MIMEMultipart('related')
    msg['Subject'] = "fup-services process error"
    msg['From'] = sender
    msg['To'] = ', '.join(destinations)
    msg['Date'] = formatdate(localtime=True)
    msg.attach(content)

    smtp = smtplib.SMTP()
    smtp.connect("smtp.blueline.mg", port=10026)
    # smtp = smtplib.SMTP('localhost')
    smtp.sendmail(sender, destinations, msg.as_string())
    smtp.quit()

    LOGS.logger.info(
        "Email: '{}' sent to {}".format(msg['Subject'], destinations))

    # message = (
    #     "Service: {} [{}]\n\n"
    #     "Affected operation: {}\n\n"
    #     "Error: {}"
    #     .format("fup-services", APIGW, operation, error)
    # )
    # destinations = ["dev@si.blueline.mg", "sysadmin@si.blueline.mg"]
    #
    # send_email(
    #     destinations=destinations,
    #     subject="fup-services process error",
    #     message=message
    # )


def send_email(destinations, subject, message, attachment=None):
    """
    Generic function for sending emails.
    :param destinations: single email or list of emails
    :param subject: email subject
    :param message: email content
    :attachment: files that will be attached to the email if any
    """
    sender = "api-gateway@si.blueline.mg"
    destinations = "yona.rasolonjatovo@staff.blueline.mg"
    if not isinstance(destinations, list):
        destinations = list(destinations)
    content = MIMEText(message, 'plain', 'utf-8')

    # if attachment:
    #     msg = MIMEMultipart('related')
    #     attachment = MIMEBase('text', 'csv')
    #     msg.attach(attachment)
    #     fp = open('/tmp/details_reconciliation.csv', 'rb')
    #     attachment.set_payload(fp.read())
    #     fp.close()
    #     attachment.add_header(
    #         'Content-Disposition',
    #         'attachment',
    #         filename='details_reconciliation.csv'
    #     )
    # else:
    #     msg = MIMEMultipart()
    msg = MIMEMultipart()
    msg['Subject'] = subject
    msg['From'] = sender
    msg['To'] = ', '.join(destinations)
    msg['Date'] = formatdate(localtime=True)
    msg.attach(content)

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(sender, destinations, msg.as_string())
    smtp.quit()

    LOGS.logger.info(
        "Email: '{}' sent to {}".format(msg['Subject'], destinations))
    smtp.quit()

# EOF
