#!/usr/bin/env python
# encoding: utf-8

"""
Ce module contient les briques de bases du service
de gestion des politiques de consommation
des utilisateurs Internet chez Blueline.

Une synchronisation des données est effectuée tous les matins
avec Aiguillier pour disposer d'une base locale afin de faire
des calculs sans craindre les ralentissements réseau. Toutes
les vérifications se font donc en local.
"""
import logging
import smtplib
from email.mime.base import MIMEBase

import requests

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate

from .consts import FUPDB_CREDENTIALS, APIGW, LOGS, CONFIG

import records
import psycopg2


__all__ = [
    'database_url',
    'record_customer',
    'apply_product_policies',
    'update_customer_cdr'
    ]


def connexion(**kwargs):
    con = psycopg2.connect(
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        database=kwargs.get('name')
    )
    return con


def database_url(**kwargs):
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=kwargs.get('engine'),
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        name=kwargs.get('name')
        )
    return url


def si_database_url():
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=CONFIG['DB']['engine'],
        user=CONFIG['DB']['user'],
        password=CONFIG['DB']['password'],
        host=CONFIG['DB']['host'],
        port=CONFIG['DB']['port'],
        name=CONFIG['DB']['name'],
    )
    return url


def get_fuped_aiguiller():
    """
    Fetches all fuped customers in Aiguiller.
    The called API returns both BI and SI customers and results need to be
    filtered accordingly.
    """
    try:
        response = requests.get(
            CONFIG['AIGUILLIER']['url'] + '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password'],
            )
        )
        return response.json()['fup']
    except Exception as e:
        operation = "get fuped customers in Aiguiller"
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def get_customers_si():
    """Fetches fup-able customers in fup-service database."""
    try:
        with records.Database(si_database_url()) as db:
            sql = ("SELECT * FROM customer")
            rows = db.query(sql)
            customer_l = rows.as_dict()
            return customer_l
    except Exception as e:
        operation = "fetch customers list in SI database"
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def get_last_history(customer_name):
    """
    Get a given customer's last fup history in fup-services database.
    :param customer_name: name of the customer, already lowered
    """
    try:
        with records.Database(si_database_url()) as db:
            sql = (
                "SELECT * FROM fup "
                "WHERE datetime = "
                "(SELECT MAX(datetime) FROM fup "
                "WHERE customer='{name}') "
            ).format(name=customer_name)
            rows = db.query(sql)
            fup_history = rows.as_dict()
        return fup_history[0]
    except Exception as e:
        operation = (
            "get customer {}'s last fup history in SI database"
            .format(customer_name)
        )
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def purge_customer(name):
    """
    Remove all information stored in database about the given customer
    """
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = "DELETE FROM customer WHERE name=:name"
        db.query(sql, name=name)
        sql = "DELETE FROM cdr WHERE customer=:name"
        db.query(sql, name=name)
        sql = "DELETE FROM fup WHERE customer=:name"
        db.query(sql, name=name)


def record_customer(infos):
    """
    Add the given customer and its information to database
    """
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = (
            "INSERT INTO customer (name, refnum, product) "
            "VALUES (:name, :refnum, :product)"
            )
        db.query(
            sql,
            name=infos['name'],
            refnum=infos['refnum'],
            product=infos['product']
            )


def apply_product_policies(for_real, customer):
    """Sous routine appliquant les politiques de bon usage
    à un client en particulier.
    Utile pour ensuite faire de l'exécution parallèle"""
    response = customer.apply_product_policies(for_real=for_real)
    data = [
        policy for policy in response
        # if 'fup_in' in policy or 'fup_out' in policy
        ]
    return customer, data


def update_customer_cdr(last_date, customer):
    """Mise à jour des CDRs d'un client dans la base REDIS"""
    customer.get_cdr_from_aiguillier(
        last_date=last_date
        )


def send_error_report(operation, error):
    """
    Sends an email to development and monitoring team to notify an error
    in the process.
    """
    message = (
        "Service: {} [{}]\n\n"
        "Affected operation: {}\n\n"
        "Error: {}"
        .format("fup-services", APIGW, operation, error)
    )
    destinations = ["dev@si.blueline.mg", "sysadmin@si.blueline.mg"]

    send_email(
        destinations=destinations,
        subject="fup-services process error",
        message=message
    )


def send_email(destinations, subject, message, attachment=None):
    """
    Generic function for sending emails.
    :param destinations: single email or list of emails
    :param subject: email subject
    :param message: email content
    :attachment: files that will be attached to the email if any
    """
    sender = "api-gateway@si.blueline.mg"
    destinations = "yona.rasolonjatovo@staff.blueline.mg"
    if not isinstance(destinations, list):
        destinations = list(destinations)
    content = MIMEText(message, 'plain', 'utf-8')

    if attachment:
        msg = MIMEMultipart('related')
        attachment = MIMEBase('text', 'csv')
        msg.attach(attachment)
        fp = open('/tmp/details_reconciliation.csv', 'rb')
        attachment.set_payload(fp.read())
        fp.close()
        attachment.add_header(
            'Content-Disposition',
            'attachment',
            filename='details_reconciliation.csv'
        )
    else:
        msg = MIMEMultipart()

    msg['Subject'] = subject
    msg['From'] = sender
    msg['To'] = ', '.join(destinations)
    msg['Date'] = formatdate(localtime=True)
    msg.attach(content)

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(sender, destinations, msg.as_string())
    smtp.quit()

    LOGS.logger.info(
        "Email: '{}' sent to {}".format(msg['Subject'], destinations))
    smtp.quit()

# EOF
