# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
from flask import jsonify, request

from blueflask.lib.errors import bad_request, not_found
from blueflask.lib.infos import no_content

from ...consts import SERVICE_CODE
from ... import customer as lib_customer
from . import api


@api.route('/customers/fup/<string:uuid>', methods=['GET'])
def get_customers_fup(uuid):
    """
    liste de client fupé
    ---
    responses:
      200:
        description: operation reussit
        $ref: "#/definitions/Informations
      500:
        description: Une erreur interne est survenue
    """
    try:
        customer = lib_customer.Customer(uuid)
    except ValueError:
        return not_found(
            message="Le client {} n'existe pas".format(uuid),
            code='{}-{}'.format(
                SERVICE_CODE,
                '4040'
                ),
            service_code=SERVICE_CODE
            )
    response = customer.customers_fup()
    return jsonify(response)


@api.route('/customers/startconsumptions/<string:uuid>', methods=['POST'])
def start_consumptions_customers(uuid):
    """
    date de debut de conso client
    ---
    tags:
      - Customers
    responses:
      200:
        description: operation reussit
        $ref: "#/definitions/Informations
      4001:
        description: Le format de date est ddmmyyyy
      4040:
        description: client n'existe pas
      500:
        description: Une erreur interne est survenue
    """
    try:
        customer = lib_customer.Customer(uuid)
        start_conso = request.args.get('start_conso', None)
        if not start_conso:
            start_conso = datetime.strftime(datetime.now(), '%d%m%Y')
        else:
            try:
                datetime.strptime(start_conso, '%d%m%Y')
            except ValueError:
                return bad_request(
                    message='Le format de date est ddmmyyyy',
                    code='{}-{}'.format(
                        SERVICE_CODE,
                        '4001'
                        ),
                    service_code=SERVICE_CODE
                    )
    except ValueError:
        return not_found(
            message="Le client {} n'existe pas".format(uuid),
            code='{}-{}'.format(
                SERVICE_CODE,
                '4040'
                ),
            service_code=SERVICE_CODE
            )
    response = customer.start_consumptions(start_conso)
    code = '{}-{}'.format(
        SERVICE_CODE,
        response['code']
        )
    response['code'] = code
    return jsonify(response)


@api.route('/customers/defup/<string:uuid>', methods=['POST'])
def defup_customers(uuid):
    """
    defupé un client
    ---
    tags:
      - Customers
    responses:
      200:
        description: operation reussit
        $ref: "#/definitions/Informations
      201:
        description: deja defupé
      204:
        description: politique du client a changé
      500:
        description: Une erreur interne est survenue
    """
    try:
        bool_defup = request.args.get('bool_defup', None)
        if not bool_defup or bool_defup.lower() == 'true':
            bool_defup = True
        elif bool_defup.lower() == 'false':
            bool_defup = False
        else:
            return not_found(
                message="Valeur possible de bool_defup true|false",
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4041'
                    ),
                service_code=SERVICE_CODE
                )
        customer = lib_customer.Customer(uuid)
    except ValueError:
        return not_found(
            message="Le client {} n'existe pas".format(uuid),
            code='{}-{}'.format(
                SERVICE_CODE,
                '4040'
                ),
            service_code=SERVICE_CODE
            )
    response = customer.defup_from_aiguillier(bool_defup)
    code = '{}-{}'.format(
        SERVICE_CODE,
        response['code']
        )
    response['code'] = code
    return jsonify(response)


@api.route('/customers', methods=['GET'])
def get_customers():
    """
    Récupération de la liste des clients.
    Cette ressource permet de récupérer la liste des clients \
    concernés par une politique de bon usage.
    ---
    tags:
      - Customers
    definitions:
      - schema:
          id: Informations
          properties:
            name:
              description: Nom d'utilisateur du client
              type: string
            refnum:
              description: Identifiant unique du client
              type: string
      - schema:
          id: Customers
          properties:
            length:
              type: integer
              description: Nombre de clients
            customers:
              type: array
              $ref: "#/definitions/Informations"
    responses:
      200:
        description: La liste des clients
        $ref: "#/definitions/Informations
      204:
        description: Aucun client n'a été trouvé
      500:
        description: Une erreur interne est survenue
    """
    customers = lib_customer.Customer.all()
    if not customers:
        return no_content(service_code=SERVICE_CODE)
    response = {
        'length': len(customers),
        'customers': [
            {
                'name': customer.name,
                'refnum': customer.refnum
                }
            for customer in customers
            ]
        }
    return jsonify(response)


@api.route('/customers/<string:uuid>', methods=['GET'])
def get_customer(uuid):
    """
    Récupération des infos d'un client.
    Cette ressource permet de récupérer les infos d'un client
    ---
    tags:
      - Customers
    definitions:
      - schema:
        id: Customer Details
        properties:
          name:
            description: Nom d'utilisateur du client
            type: string
          refnum:
            description: Identifiant unique du client
            type: string
          product:
            description: Nom du produit du client
            type: string
          fup_status:
            description: Etat du client par politique
            type: array
            items:
              properties:
                status:
                  type: string
                  description: Etat du client
                interval:
                  type: string
                  description: Interval horaire d'application de la politique
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: Les informations du client
      404:
        description: Aucun client n'a été trouvé
      500:
        description: Une erreur interne est survenue
    """
    try:
        int(uuid)
    except ValueError:
        customer = lib_customer.Customer(uuid)
    else:
        search_customer = [
            customer for customer in lib_customer.Customer.all()
            if customer.refnum == uuid
            ]
        if search_customer:
            customer = search_customer.pop()
        else:  # aucun client n'a été trouvé
            return not_found(
                message="Le client {} n'existe pas".format(uuid),
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4040'
                    ),
                service_code=SERVICE_CODE
                )
    response = {
        'refnum': customer.refnum,
        'name': customer.name,
        'product': customer.product.name,
        'fup_status': customer.fup_status
        }
    return jsonify(response)


@api.route('/customers/consumptions/<string:uuid>', methods=['GET'])
def get_customer_consumption(uuid):
    """
    Récupération de la consommation d'un client.
    Cette ressource permet d'avoir la consommation détaillée \
    d'un client sur une période donnée.
    ---
    tags:
      - Customers
    responses:
      200:
        description: La consommation détaillée
        schema:
          id: Détails de consommation
          properties:
            name:
              description: Nom d'utilisateur du client
              type: string
            refnum:
              description: Identifiant unique du client
              type: string
            consumption:
              description: Consommation du client
              properties:
                start:
                  description: Date de début
                stop:
                  description: Date de fin
                details:
                  description: détails
                  type: array
                  items:
                    properties:
                      date:
                        description: Date de consommation
                      total
                        description: Total consommation en octets
                      octets_in
                        description: Total téléchargement en octets
                      octets_out
                        description: Total téléversement en octets
                      policy_wise
                        description: Consommation suivant les politiques
                        type: array
                        items:
                          properties:
                            data_limit: Limite autorisée en Go
                            start: Début d'application de la politique
                            stop: Fin d'application de la politique
                            total: Total consommation en octets
                            octets_in: Total téléchargement en octets
                            octets_out: Total téléversement en octets
      204:
        description: Aucune donnée de consommation trouvée
      404:
        description: Le client n'existe pas
      500:
        description: Erreur interne
    """
    try:
        product_out_fup = request.args.get('product_out_fup', None)
        if product_out_fup:
            if product_out_fup.lower() == 'true':
                kwargs = {
                    'table_customer': 'customer_ese',
                    'table_cdr': 'cdr_ese'
                }
                customer = lib_customer.Customer(uuid, **kwargs)
        else:
            product_out_fup = False
            customer = lib_customer.Customer(uuid)
    except ValueError:
        return not_found(
            message="Le client {} n'existe pas".format(uuid),
            code='{}-{}'.format(
                SERVICE_CODE,
                '4040'
                ),
            service_code=SERVICE_CODE
            )
    else:
        bool_start = request.args.get('bool_start', False)
        start = request.args.get('start', None)
        stop = request.args.get('stop', None)
        if bool_start:
            if bool_start.lower() == 'true':
                bool_start = True
            else:
                bool_start = False
        if not start:
            start = datetime.now() - timedelta(30)
        else:
            try:
                start = datetime.strptime(start, '%d%m%Y')
            except ValueError:
                return bad_request(
                    message='Le format de date est ddmmyyyy',
                    code='{}-{}'.format(
                        SERVICE_CODE,
                        '4001'
                        ),
                    service_code=SERVICE_CODE
                    )
        if not stop:
            stop = datetime.now() - timedelta(1)
        else:
            try:
                stop = datetime.strptime(stop, '%d%m%Y')
            except ValueError:
                return bad_request(
                    message='Le format correct de date est ddmmyyyy',
                    code='{}-{}'.format(
                        SERVICE_CODE,
                        '4001'
                        ),
                    service_code=SERVICE_CODE
                    )
        infos = customer.get_cdr_summary(
            start.date(),
            stop.date(),
            True,
            product_out_fup,
            bool_start
            )
        return jsonify(infos)

# EOF
