#!/usr/bin/env python
# encoding: utf-8


from functools import partial
from multiprocessing.pool import Pool
from random import sample

from .consts import CONFIG
from .consts import FUPDB_CREDENTIALS
from .consts import LOGS
from .consts import REST_CLIENT
from .consts import APIGW
from . import core

import records
import requests


__all__ = [
    'Product'
    ]


class Product(object):
    """Toute politique de bon usage est centrée autour d'un produit.
    Un produit est identifié par son nom, et ses politiques de bon usage"""

    def __init__(self, name):
        self.name = name
        self.policies = {}
        policies_names = CONFIG['POLICIES']
        policy_names = [
            policy_name for policy_name in policies_names
            if policy_name.startswith(self.name)
            ]
        for policy_name in policy_names:
            data_limit, interval = policies_names[policy_name].split(',', 1)
            self.policies[policy_name] = {
                'data_limit': int(data_limit),
                'interval': interval
                }
        self.no_policies = {}
        no_policies_names = CONFIG['NO_POLICIES']
        no_policy_names = [
            no_policy_name for no_policy_name in no_policies_names
            if no_policy_name.startswith(self.name)
            ]
        for no_policy_name in no_policy_names:
            data_limit, interval = no_policies_names[no_policy_name].split(
                ',',
                1
                )
            name_type = no_policy_name.split('_')
            self.no_policies[no_policy_name] = {
                'data_limit': int(data_limit),
                'interval': interval,
                'name_type': name_type[len(name_type)-2]
                }

    def __repr__(self):
        return '<Product {name}>'.format(name=self.name)

    @property
    def customers(self):
        """Récupération de la liste des clients possédant ce produit"""
        from .customer import Customer
        customers = [
            c for c in Customer.all()
            if c.product.name == self.name
            ]
        return customers

    @property
    def customers_ese(self):
        """Récupération de la liste des clients possédant ce produit"""
        from .customer import Customer
        customers = [
            c for c in Customer.all_ese()
            if c.product.name == self.name
            ]
        return customers

    @staticmethod
    def all():
        policy_names = [i for i in CONFIG['POLICIES']]
        products = set(['_'.join(i.split('_')[:-1]) for i in policy_names])
        return [Product(name) for name in products]

    @staticmethod
    def all_ese():
        policy_names = [i for i in CONFIG['NO_POLICIES']]
        products = set(['_'.join(i.split('_')[:-1]) for i in policy_names])
        return [Product(name) for name in products]

    def apply_policies(self, for_real=False, random=False):
        """Application des politiques de bon usage à tous les clients
        abonnés à ce produit."""

        customers = self.customers
        if random:
            customers = sample(customers, 3)
        f = partial(core.apply_product_policies, for_real)
        with Pool(4) as p:
            response = p.map(f, customers)
        customers = {i[0].name: i[1] for i in response if i[1]}
        policies = []
        for policy_name, policy in self.policies.items():
            policy_infos = {
                'data_limit': policy['data_limit'],
                'name': policy_name,
                'interval': policy['interval'],
                'customers': {}
                }
            policy_customers = policy_infos['customers']
            for customer_id, data in customers.items():
                for info in data:
                    if info['name'] == policy_name:
                        policy_customers[customer_id] = {
                            'total': info['total'],
                            'octets_in': info['octets_in'],
                            'octets_out': info['octets_out'],
                            'sources': info['sources'],
                            'details': info['details']
                            }
                        if 'fup_in' in info:
                            policy_customers[customer_id]['fup_in'] = True
                        elif 'fup_out' in info:
                            policy_customers[customer_id]['fup_out'] = True
            policies.append(policy_infos)
        return policies

    def get_customers_aiguillier(self):
        """Récupération de la liste de clients depuis aiguillier"""
        response = REST_CLIENT.get(
            '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password']
                ),
            params={'product': self.name}
            )
        customers = response['content'].get('customers', [])
        customers = {
            customer['name'].lower(): {
                'product': customer['product'],
                'refnum': customer['refnum']
                }
            for customer in customers
            }
        return customers

    def get_customers_bluebase(self, username='fup-services'):
        """Récupération de la liste de clients depuis Bluebase"""
        LOGS.logger.info(self.name)
        customers = {}
        bluebase_url = (
            'https://{fqdn}/bluebase'
            '/v1/fup/tech_business_list'
            ).format(
                fqdn=APIGW
                )
        response = requests.post(
            url=bluebase_url,
            auth=(username, ''),
            json={'type': self.name}
            )
        if response.status_code == 200:
            response = response.json()
            data = response['root']['data']
            infos = data['list']['produit']
            if isinstance(infos, dict):
                infos = [infos]
            customers = {
                info['ident'].lower(): {
                    'product': info['produit_nom'].lower(),
                    'refnum': info['refnum_primaire']
                    }
                for info in infos
                }
        return customers

    def get_customers_gateway(self):
        """Récupération de la liste de clients depuis api"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "SELECT * FROM customer "
                "WHERE product=:product"
                )
            rows = db.query(
                sql,
                product=self.name
                )
            customers = {
                customer['name'].lower(): {
                    'product': customer['product'],
                    'refnum': customer['refnum']
                    }
                for customer in rows
                }
        return customers

    def update_customers(self, product_ese=False):
        """Récupération de la nouvelle liste de clients depuis aiguillier"""
        from .customer import Customer
        customers = self.get_customers_aiguillier()
        url = core.database_url(**FUPDB_CREDENTIALS)
        if not product_ese:
            LOGS.logger.info("update customer")
            with records.Database(url) as db:
                sql = "DELETE FROM customer WHERE product=:product"
                db.query(sql, product=str(self.name))
            existing_customers = {
                customer.name: {
                    'product': customer.product,
                    'refnum': customer.refnum
                    }
                for customer in self.customers
                }
            # existing_customers c'est la liste des clients
            # que l'application classe comme possédant le produit
            for customer, infos in customers.items():
                existing_customer = existing_customers.get(customer)
                if existing_customer:
                    Customer(customer).set_product(
                        infos['product']
                        )
                    # on vérifie le refnum du client a changé
                    if existing_customer['refnum'] != infos['refnum']:
                        # on met à jour le refnum
                        Customer(customer).set_refnum(
                            infos['refnum']
                            )
                else:
                    # soit le client 'customer' n'existe pas
                    # soit il n'a pas le produit 'self.name'
                    try:
                        # est-ce-que le client existe dans notre base ?
                        c = Customer(customer)
                    except ValueError:
                        # le client n'existe vraiment pas dans la base
                        # on va le créer
                        # url = core.database_url(**FUPDB_CREDENTIALS)
                        with records.Database(url) as db:
                            sql = (
                                "INSERT INTO customer (name, refnum, product) "
                                "VALUES (:name, :refnum, :product)"
                                )
                            LOGS.logger.info(sql)
                            db.query(
                                sql,
                                name=customer,
                                refnum=infos['refnum'],
                                product=infos['product']
                                )
                        LOGS.logger.info(
                            "Inscription du client {} "
                            "in table customer".format(
                                customer
                                )
                            )
                    else:
                        # le client existe déjà dans la base
                        # donc c'est forcément son produit qui a changé
                        # on modifie d'abord le refnum au cas où
                        c = Customer(customer)
                        # modification du refnum au cas où
                        c.set_refnum(infos['refnum'])
                        current_product = c.product
                        c.set_product(infos['product'])
                        # on défup de l'ancien s'il faut
                        customer_status = c.fup_status
                        for policy_name in current_product.policies:
                            status = customer_status.get(policy_name)
                            if status:
                                status = sorted(
                                    status,
                                    key=lambda x: x['datetime'],
                                    reverse=True
                                    )
                                if status[0]['status']:
                                    # si le client était fup, il faut le défup
                                    REST_CLIENT.delete(
                                        '/fup',
                                        auth=(
                                            CONFIG['AIGUILLIER']['username'],
                                            CONFIG['AIGUILLIER']['password']
                                            ),
                                        params={'customer': self.name}
                                        )
        else:
            kwargs = {
                'table_customer': 'customer_ese',
                'table_cdr': 'cdr_ese'
            }
            with records.Database(url) as db:
                sql = "DELETE FROM customer_ese WHERE product=:product"
                db.query(sql, product=str(self.name))
            for customer, infos in customers.items():
                # soit le client 'customer' n'existe pas
                # soit il n'a pas le produit 'self.name'
                try:
                    # est-ce-que le client existe dans notre base ?
                    c = Customer(customer, **kwargs)
                except ValueError:
                    # le client n'existe vraiment pas dans la base
                    # on va le créer
                    # url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO customer_ese (name, refnum, "
                            "product) VALUES (:name, :refnum, :product)"
                            )
                        LOGS.logger.info(sql)
                        db.query(
                            sql,
                            name=customer,
                            refnum=infos['refnum'],
                            product=infos['product']
                            )
                    LOGS.logger.info(
                        "Inscription du client {} "
                        "in table customer_ese".format(
                            customer
                            )
                        )
                else:
                    # le client existe déjà dans la base
                    # donc c'est forcément son produit qui a changé
                    # on modifie d'abord le refnum au cas où
                    c = Customer(customer, **kwargs)
                    # modification du refnum au cas où
                    c.set_refnum(infos['refnum'])
                    current_product = c.product
                    c.set_product(infos['product'])

# EOF
