#!/usr/bin/env python
# encoding: utf-8

from email.mime.multipart import MIMEMultipart
from email.mime.base import MIMEBase
from email.mime.text import MIMEText
from multiprocessing.pool import Pool

from .. import CONFIG
from .. import core
from ..customer import Customer
from ..product import Product
from ..product_gp import Product_GP
from . import app

import click
import smtplib
import tablib


@click.group()
def cmd():
    pass


@cmd.command()
@click.option('--host', default='0.0.0.0')
@click.option('--port', default=5000)
def runserver(host, port):
    """Lancer le serveur Werkzeug pour l'API REST (dev. seulement)"""
    app.run(host=host, port=port, debug=True)


@cmd.command()
def sync():
    """Mise à jour de la liste des clients Business ainsi que de leurs \
    relevés de consommation des 30 derniers jours."""
    products = Product.all()
    for product in products:
        product.update_customers()
        with Pool(2) as pool:
            pool.map(core.update_customer_cdr, product.customers)


@cmd.command()
def notify():
    """Commande pour notifier les clients grand public \
    selon leur niveau de consommation"""
    click.echo('Récupération des données depuis Bluebase')
    customers = Product_GP.all()
    for customer in customers:
        click.echo('Notification du client {}'.format(customer.msisdn))
        response = customer.notify()
        if not response:
            click.echo("Pas de notification à envoyer")
        else:
            medium, percentage = response
            click.echo(
                '{}: {}% (Notification envoyée)'.format(
                    medium,
                    percentage
                    )
                )


@cmd.command()
@click.option('--append-details', is_flag=True, default=False)
@click.option('--for-real', is_flag=True, default=False)
@click.option('--send-mail', is_flag=True, default=False)
def apply(append_details, for_real, send_mail):
    """Appliquer les politiques de FUP pour tous les clients Business"""
    result = {}
    products = Product.all()
    for product in products:
        policies = product.apply_policies(for_real=for_real)
        if policies:
            result[product.name] = policies
    if result:
        message = ''
        dataset = tablib.Dataset()
        dataset.headers = [
            'product',
            'data_limit',
            'policy_name',
            'customer',
            'download',
            'upload',
            'total',
            'status'
            ]
        products = sorted([i for i in result], reverse=True)
        for product in products:
            policies = result[product]
            message += 'Produit: {product}\n'.format(product=product)
            for policy in policies:
                message += (
                    '\n'
                    'Limite autorisée: {data_limit} Go\n'
                    ).format(
                        data_limit=policy['data_limit']
                        )
                foo = ''
                for customer, data in policy['customers'].items():
                    customer_policies_status = Customer(customer).fup_status
                    policy_status = False
                    for customer_policy_status in customer_policies_status:
                        if customer_policy_status['policy'] == policy['name']:
                            policy_status = customer_policy_status['status']
                            break
                    dataset.append(
                        [
                            product,
                            policy['data_limit'],
                            policy['name'],
                            customer,
                            data['octets_in'],
                            data['octets_out'],
                            data['total'],
                            policy_status
                            ]
                        )
                    if data.get('fup_in'):
                        status = 'fup_in'
                    elif data.get('fup_out'):
                        status = 'fup_out'
                    else:
                        status = None
                    if status:
                        foo += (
                            ' - {customer}: {total} Go ({status})\n'.format(
                                customer=customer,
                                total=data['total'],
                                status=status
                                )
                            )
                if foo:
                    message += foo
                else:
                    message += "Aucun client à FUPer\n"
            message += '\n\n\n'
        click.echo(message)
        with open('/tmp/details.csv', 'w') as f:
            f.write(dataset.export('csv'))
        if send_mail:
            recipients = CONFIG['ALERT']['to']
            content = MIMEText(message, 'plain', 'utf-8')
            attachment = MIMEBase('text', 'csv')
            fp = open('/tmp/details.csv', 'rb')
            attachment.set_payload(fp.read())
            fp.close()
            attachment.add_header(
                'Content-Disposition',
                'attachment',
                filename='details.csv'
                )
            msg = MIMEMultipart()
            msg['Subject'] = CONFIG['ALERT']['subject']
            msg['From'] = CONFIG['ALERT']['from']
            msg['To'] = recipients
            msg.attach(attachment)
            msg.attach(content)
            s = smtplib.SMTP('localhost')
            s.sendmail(
                CONFIG['ALERT']['from'],
                recipients.replace(' ', '').split(','),
                msg.as_string()
                )
            s.quit()

# EOF
