#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime, timedelta
from flask import jsonify, request

from blueflask.lib.decorators import auth as ldap
from blueflask.lib.errors import bad_request, not_found
from blueflask.lib.infos import no_content

from ... import SERVICE_CODE
from ... import customer as lib_customer
from . import api


@api.route('/customers', methods=['GET'])
@ldap.login_required
def get_customers():
    """
    Récupération de la liste des clients.
    Cette ressource permet de récupérer la liste des clients \
    concernés par une politique de bon usage.
    ---
    tags:
      - Customers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: La liste des clients
        schema:
          id: Liste de clients
          properties:
            length:
              type: integer
              description: Nombre total de clients trouvés
            customers:
              type: array
              items:
                properties:
                  name:
                    description: Nom d'utilisateur du client
                    type: string
                  refnum:
                    description: Identifiant unique du client
                    type: string
      204:
        description: Aucun client n'a été trouvé
      500:
        description: Une erreur interne est survenue
    """
    customers = lib_customer.Customer.all()
    if not customers:
        return no_content(service_code=SERVICE_CODE)
    response = {
        'length': len(customers),
        'customers': [
            {
                'name': customer.name,
                'refnum': customer.refnum
                }
            for customer in customers
            ]
        }
    return jsonify(response)


@api.route('/customers/<string:uuid>', methods=['GET'])
@ldap.login_required
def get_customer(uuid):
    """
    Récupération des infos d'un client.
    Cette ressource permet de récupérer les infos d'un client
    ---
    tags:
      - Customers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: Les informations du client
        schema:
          id: Informations du client
          properties:
            name:
              description: Nom d'utilisateur du client
              type: string
            refnum:
              description: Identifiant unique du client
              type: string
            product:
              description: Nom du produit du client
              type: string
            fup_status:
              description: Etat du client par politique
              type: array
              items:
                properties:
                  status:
                    type: string
                    description: Etat du client
                  interval:
                    type: string
                    description: Interval horaire d'application de la politique
      404:
        description: Aucun client n'a été trouvé
      500:
        description: Une erreur interne est survenue
    """
    try:
        int(uuid)
    except ValueError:
        customer = lib_customer.Customer(uuid)
    else:
        search_customer = [
            customer for customer in lib_customer.Customer.all()
            if customer.refnum == uuid
            ]
        if search_customer:
            customer = search_customer.pop()
        else:  # aucun client n'a été trouvé
            return not_found(
                message="Le client {} n'existe pas".format(uuid),
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4040'
                    ),
                service_code=SERVICE_CODE
                )
    response = {
        'refnum': customer.refnum,
        'name': customer.name,
        'product': customer.product.name,
        'fup_status': customer.fup_status
        }
    return jsonify(response)


@api.route('/customers/consumptions/<string:uuid>', methods=['GET'])
@ldap.login_required
def get_customer_consumption(uuid):
    """
    Récupération de la consommation d'un client.
    Cette ressource permet d'avoir la consommation détaillée \
    d'un client sur une période donnée.
    ---
    tags:
      - Customers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: La consommation détaillée
        schema:
          id: Détails de consommation
          properties:
            name:
              description: Nom d'utilisateur du client
              type: string
            refnum:
              description: Identifiant unique du client
              type: string
            consumption:
              description: Consommation du client
              properties:
                start:
                  description: Date de début
                stop:
                  description: Date de fin
                details:
                  description: détails
                  type: array
                  items:
                    properties:
                      date:
                        description: Date de consommation
                      total
                        description: Total consommation en Go
                      octets_in
                        description: Total téléchargement en Go
                      octets_out
                        description: Total téléversement en Go
                      policy_wise
                        description: Consommation suivant les politiques
                        type: array
                        items:
                          properties:
                            data_limit: Limite autorisée en Go
                            start: Début d'application de la politique
                            stop: Fin d'application de la politique
                            total: Total consommation en Go
                            octets_in: Total téléchargement en Go
                            octets_out: Total téléversement en Go
      204:
        description: Aucune donnée de consommation trouvée
      404:
        description: Le client n'existe pas
      500:
        description: Erreur interne
    """
    try:
        int(uuid)
    except ValueError:
        customer = lib_customer.Customer(uuid)
    else:
        search_customer = [
            customer for customer in lib_customer.Customer.all()
            if customer.refnum == uuid
            ]
        if search_customer:
            customer = search_customer.pop()
        else:  # aucun client n'a été trouvé
            return not_found(
                message="Le client {} n'existe pas".format(uuid),
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4040'
                    ),
                service_code=SERVICE_CODE
                )
    start = request.args.get('start', None)
    stop = request.args.get('stop', None)
    if not start:
        start = datetime.now() - timedelta(30)
    else:
        try:
            start = datetime.strptime(start, '%d%m%Y')
        except ValueError:
            return bad_request(
                message='Le format de date est ddmmyyyy',
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4001'
                    ),
                service_code=SERVICE_CODE
                )
    if not stop:
        stop = datetime.now() - timedelta(1)
    else:
        try:
            stop = datetime.strptime(stop, '%d%m%Y')
        except ValueError:
            return bad_request(
                message='Le format correct de date est ddmmyyyy',
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4001'
                    ),
                service_code=SERVICE_CODE
                )
    infos = customer.get_cdr_summary(start.date(), stop.date())
    return jsonify(infos)

# EOF
