# -*- coding: utf-8 -*-

from datetime import datetime

from flask import jsonify, request

from blueflask.lib.errors import bad_request
from blueflask.lib.infos import success

from ...consts import SERVICE_CODE
from ... import card_management
from . import api


@api.route('/cards/<string:cardnumber>', methods=['DELETE'])
def destroy_card(cardnumber):
    """
    Suppression d'une carte.
    ---
    tags:
       - Card
    """
    result = card_management.blacklist_card(cardnumber)
    response = jsonify(result)
    response.status_code = result['status']
    return response


@api.route('/cards', methods=['DELETE'])
def destroy_cards():
    """
    Suppression de plusieurs cartes.
    ---
    tags:
       - Cards
    """
    data = request.get_json()
    try:
        cardnumbers = data['cardnumbers']
    except KeyError:
        return bad_request(
            message='Vous devez préciser les numéros de cartes à supprimer',
            service_code=SERVICE_CODE
            )
    else:
        result = [
            card_management.blacklist_card(cardnumber)
            for cardnumber in cardnumbers
            ]
        messages = []
        for info in result:
            msg = info['message']
            msg['code'] = info['code']
            messages.append(msg)
        response = success(
            message=messages,
            service_code=SERVICE_CODE
            )
        return response


@api.route('/cards/subscriptions/<string:cardnumber>', methods=['POST'])
def add_subscription_to_card(cardnumber):
    """
    Ajout d'un nouvel abonnement à une carte.
    ---
    tags:
      - Card
    """
    data = request.get_json()
    try:
        package = data['package']
    except KeyError:
        return bad_request(message='Il faut préciser le numéro du bouquet')
    else:
        start = data.get('start', datetime.now().date().strftime('%Y-%m-%d'))
        stop = data.get('stop', '2030-12-31')
        result = card_management.add_subscription(
            cardnumber,
            start,
            stop,
            package
            )
        response = jsonify(result)
        response.status_code = result['status']
        return response


@api.route('/cards/subscriptions', methods=['POST'])
def add_subscription_to_cards():
    """
    Ajout de plusieurs abonnements
    ---
    tags:
      - Cards
    """
    data = request.get_json()
    try:
        subscriptions = data['subscriptions']
    except KeyError:
        return bad_request(
            message='Il faut préciser les abonnements à faire',
            service_code=SERVICE_CODE
            )
    else:
        messages = []
        for subscription in subscriptions:
            try:
                cardnumber = subscription['cardnumber']
                package = subscription['package']
            except KeyError:
                return bad_request(
                    message='Il faut préciser les cartes et les bouquets',
                    service_code=SERVICE_CODE
                    )
            else:
                start = subscription.get(
                    'start',
                    datetime.now().date().strftime('%Y-%m-%d')
                    )
                stop = subscription.get(
                    'stop',
                    datetime.now().date().strftime('%Y-%m-%d')
                    )
                result = card_management.add_subscription(
                    cardnumber,
                    start,
                    stop,
                    package
                    )
                msg = result['message']
                msg['code'] = result['code']
                messages.append(msg)
        response = success(
            message=messages,
            service_code=SERVICE_CODE
            )
        return response


@api.route('/cards/subscriptions/<string:cardnumber>', methods=['PUT'])
def stop_card_subscription(cardnumber):
    """
    Retrait d'un abonnement d'une carte.
    ---
    tags:
      - Card
    """
    data = request.get_json()
    try:
        package = data['package']
    except KeyError:
        return bad_request(message='Il faut préciser le numéro du bouquet')
    else:
        date = data.get('date', datetime.now().date().strftime('%Y-%m-%d'))
        result = card_management.stop_subscription(
            cardnumber,
            date,
            package
            )
        response = jsonify(result)
        response.status_code = result['status']
        return response


@api.route('/cards/subscriptions', methods=['PUT'])
def stop_cards_subscriptions():
    """
    Retrait de plusieurs abonnements
    ---
    tags:
      - Cards
    """
    data = request.get_json()
    try:
        subscriptions = data['subscripptions']
    except KeyError:
        return bad_request(
            message='Il faut préciser les abonnements à faire',
            service_code=SERVICE_CODE
            )
    else:
        messages = []
        for subscription in subscriptions:
            try:
                cardnumber = subscription['cardnumber']
                package = subscription['package']
            except KeyError:
                return bad_request(
                    message='Il faut préciser les cartes et les bouquets',
                    service_code=SERVICE_CODE
                    )
            else:
                date = subscription.get(
                    'date',
                    datetime.now().date().strftime('%Y-%m-%d')
                    )
                result = card_management.stop_subscription(
                    cardnumber,
                    date,
                    package
                    )
                msg = result['message']
                msg['code'] = result['code']
                messages.append(msg)
        response = success(
            message=messages,
            service_code=SERVICE_CODE
            )
        return response


@api.route('/cards/<string:cardnumber>', methods=['GET'])
def get_card_informations(cardnumber):
    """
    Récupération des informations d'une carte TV.
    Cette ressource permet de récupérer les informations
    d'abonnement, d'état et d'association d'une carte TV.
    ---
    tags:
      - Cards
    responses:
      200:
        description: Informations sur la carte
    """
    result = card_management.get_card_informations(cardnumber)
    response = jsonify(result)
    response.status_code = result['status']
    return response


@api.route('/cards', methods=['GET'])
def get_cards_informations():
    """
    Récupération des informations de plusieurs cartes TV.
    Cette ressource permet de récupérer les informations
    d'abonnement, d'état et d'association.
    ---
    tags:
      - Cards
    responses:
      200:
        description: Informations sur les cartes
    """
    data = request.get_json()
    try:
        cardnumbers = data['cardnumbers']
    except KeyError:
        return bad_request(message="Il faut préciser les numéros de cartes")
    else:
        messages = []
        for cardnumber in cardnumbers:
            result = card_management.get_card_informations(cardnumber)
            msg = result['message']
            msg['code'] = result['code']
            messages.append(msg)
        response = success(
            message=messages,
            service_code=SERVICE_CODE
            )
        return response


@api.route('/cards/<string:cardnumber>', methods=['POST'])
def set_card_status(cardnumber):
    """
    Modification de l'état actif/inactif d'une carte.
    Cette ressource permet de mettre une carte en mode
    actif ou inactif
    ---
    tags:
      - Card
    responses:
      200:
        description: Action réussie
    """
    data = request.get_json()
    if 'status' not in data or data['status'] not in ['on', 'off']:
        return bad_request(service_code=SERVICE_CODE)
    else:
        if data['status'] == 'on':
            status = True
        else:
            status = False
        result = card_management.set_card_status(cardnumber, status)
        response = jsonify(result)
        response.status_code = result['status']
        return response


@api.route('/cards', methods=['POST'])
def set_cards_status():
    """
    Modification de l'état actif/inactif de plusieurs cartes.
    ---
    tags:
      - Cards
    responses:
      200:
        description: Action réussie
    """
    data = request.get_json()
    if 'statuses' not in data:
        return bad_request(service_code=SERVICE_CODE)
    else:
        statuses = data['statuses']
        messages = []
        for infos in statuses:
            request_is_bad = (
                'cardnumber' not in infos or
                'status' not in infos or
                infos['status'] not in ['on', 'off']
                )
            if request_is_bad:
                return bad_request(service_code=SERVICE_CODE)
            else:
                cardnumber = infos['cardnumber']
                status = infos['status']
                if infos['status'] == 'on':
                    status = True
                else:
                    status = False
                result = card_management.set_card_status(cardnumber, status)
                msg = result['message']
                msg['code'] = result['code']
                messages.append(msg)
        response = success(
            message=messages,
            service_code=SERVICE_CODE
            )
        return response


@api.route('/cards/<string:cardnumber>', methods=['PUT'])
def set_card_pairing(cardnumber):
    """
    Modification de l'association d'une carte avec un STB.
    ---
    tags:
      - Card
    """
    data = request.get_json()
    if not data or 'cardnumber' not in data:
        return bad_request(service_code=SERVICE_CODE)
    elif 'stb' in data:  # on associe la carte à cette STB
        result = card_management.pair_card(data['cardnumber'], data['stb'])
    else:
        result = card_management.un_pair_card(data['cardnumber'])
    response = jsonify(result)
    response.status_code = result['status']
    return response


@api.route('/cards', methods=['PUT'])
def set_cards_pairing(cardnumber):
    """
    Modification de l'association des cartes
    ---
    tags:
      - Cards
    """
    data = request.get_json()
    if 'pairings' not in data:
        return bad_request(service_code=SERVICE_CODE)
    else:
        pairings = data['pairings']
        messages = []
        for infos in pairings:
            if 'cardnumber' not in infos:
                return bad_request(service_code=SERVICE_CODE)
            else:
                cardnumber = infos['cardnumber']
                stb = infos.get('stb', None)
                if stb:  # on associe la carte à cette STB
                    result = card_management.pair_card(cardnumber, stb)
                else:
                    result = card_management.un_pair_card(cardnumber)
                msg = result['message']
                msg['code'] = result['code']
                messages.append(msg)
        response = success(
            message=messages,
            service_code=SERVICE_CODE
            )
        return response

# EOF
