# blueflask

Ce package est une collection de fonctions, applications, décorateurs réutilisables dans le cadre d'applications Flask.

Il est écrit en python >= 3.4 et utilise Flask et quelques extensions déjà bien célèbres.

## Installation

```shell
$ pip install --index http://miroir.blueline.mg/pypi blueflask
```

## Librairies incluses

### `blueflask.lib.errors`

Ce module contient des fonctions qui retournent des objets flask.Response correspondant à des erreurs HTTP, suivant le schéma ci-dessous:

```json
{
    'error': 'some http error',
	'code': 'an_error_code_for_the_app',
    'status': an_http_error_code,
    'message': 'a message explaining the error'
}
```

Les erreurs répertoriées sont:

 * `internal_error()`
 * `bad_request()`
 * `unauthorized()`
 * `forbidden()`
 * `not_found()`
 * `not_allowed()`

```python

from flask import Flask, jsonify
from blueflask.lib.errors import not_found


app = Flask(__name__)


users = {
   'john': 'John Doe',
   'bob': 'Bob Rakoto'
    }


@app.route('/users/<string:name>', methods=['GET'])
def get_user(name):
    if name not in users:
        message = 'The user {} does not exist'.format(name)
        return not_found(message)
    else:
        response = {
            'name': 'John Doe',
            'nickname': 'John'
            }
        return jsonify(response)
```

### `blueflask.lib.infos`

Ce module contient des fonctions qui retournent des objets flask.Response correspondant à des infos HTTP (codes 2xx, 3xx), suivant le schéma ci-dessous:

```json
{
    'info': 'some http info',
    'status': an_http_info_code,
    'message': 'a message explaining the info'
}
```

Les infos répertoriées sont:

 * `no_content()`
 * `not_modified()`
 * `created()`

```python

from flask import Flask, jsonify
from blueflask.infos import not_modified, no_content, created


app = Flask(__name__)


users = {
   'john': 'John Doe',
   'bob': 'Bob Rakoto'
    }


@app.route('/users', methods=['POST'])
def create_user():
    name = request.form['name']
    nickname = request.form['nickname']
    if users.get(nickname) == name:
        return not_modified()
    elif nickname in users:
        users[nickname] = name
        return no_content()
    else:
        users[nickname] = name
        return created()
```

### `blueflask.flask_api`

Cette librairie contient une application Flask standard, réutilisable et extensible.

La particularité de cette application est:
   
 * de créer des logs avant et après chaque requête
 
 * d'ajouter au besoin un RequestID dans l'en-tête de la requête
 
 * de transformer les _docstrings_ de vos différents endpoints en documentation OpenAPI (pour peu que ces _docstrings_ respectent le format
   utilisé par flask-swagger)

 * de servir la documentation générée en utilisant Swagger UI (via flask-swagger-ui)
 

Exemple:

```python
#!/usr/bin/env python
# encoding: utf-8


from flask import jsonify

from blueflask.flask_api import FlaskApp
from blueflask.lib import infos

import random

flask_app = FlaskApp('un_service_code')
flask_app.produce_handlers_and_documentation()

app = flask_app.app
app.config['TITLE'] = 'Mon API'
app.config['DESCRIPTION'] = "API pour faire des tests"
app.config['VERSION'] = '1.0'


@app.route('/greetings', methods=['GET'])
def say_hello():
    """
	Ressource permettant de saluer le monde.
	Cette ressource permet de renvoyer tout simplement un 'Hello, world'
	---
	tags:
	  - Index
	definitions:
	  - schema:
	      id: Greeting
		  properties:
		    status
              type: number
              description: Status de la requête HTTP
            code:
              type: string
              description: Code interne à l'API
            info:
              type: string
              description: Message court décrivant l'info
		    message:
			  description: Message de salutation
			  type: string
	responses:
	  200:
	    description: Message poli de salutation
		$ref: "#/definitions/Greeting"
	"""	
    greetings = [
	    'Hallo',
		'Howzit?',
        '¡Hola!',
        'Hello',
        'Bonjour',
        'Güten tag',
        'Zdravstui',
		'Bom dia'
        ]
    return infos.success(random.choice(greetings))

if __name__ == '__main__':
    app.run(host='127.0.0.1', port=5000, debug=True)
```

Si vous lancez cet exemple, vous aurez alors accès à 3 URLs:

 * http://127.0.0.1:5000/api/spec : la spécification OpenAPI de votre API
 * http://127.0.0.1:5000/greetings : votre _endpoint_ que vous venez d'écrire
 * http://127.0.0.1:5000/apidoc : la documentation affichée avec Swagger UI
